package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.annotations.SkipEncoding;


/**
 * Class describing a muted alert request. This class is published on 
 * on the buses for the consumption of ccs-shells (via the toString()) method to
 * get the id of a request, and by ccs-consoles for identifying which alerts are
 * muted in the alert viewer.
 * 
 * A List of active MutedAlertRequest instances is published by the MMM in a
 * StatusSubsystemData with key MutedAlertRequest.PUBLICATION_KEY
 *
 */
@SkipEncoding
public class MutedAlertRequest implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 54423487100235214L;
    
    public static final String PUBLICATION_KEY = "MutedAlertRequestPublicationKey";
    
    private static final AtomicLong idCreator = new AtomicLong(0);
    private final long uniqueId;
    private final String subsystemName;
    private final String userName;
    private final long creationMillis;
    private final int durationInSeconds;
    private final Pattern alertIdPattern;
    private final String alertIdRegEx;

    /**
     * 
     * @param subsystemName
     * @param alertIdRegEx
     * @param durationInSeconds
     * @param userName 
     */
    public MutedAlertRequest(String subsystemName, String alertIdRegEx, int durationInSeconds, String userName) {
        this.uniqueId = idCreator.getAndAdd(1L);
        this.creationMillis = System.currentTimeMillis();
        this.subsystemName = subsystemName;
        this.alertIdPattern = Pattern.compile(alertIdRegEx);
        this.durationInSeconds = durationInSeconds;
        this.userName = userName;
        this.alertIdRegEx = alertIdRegEx;
    }

    /**
     * Get the unique Id for this MutedAlertRequest. This id can be used
     * to cancel this request and un-mute immediately the affected alerts.
     * 
     * @return The unique id.
     */
    public long getUniqueId() {
        return uniqueId;
    }

    /**
     * The name of the subsystem affected by this MutedAlertRequest.
     * 
     * @return The subsystem name.
     */
    public String getSubsystemName() {
        return subsystemName;
    }

    /**
     * The username that requested the muting of the alerts.
     * 
     * @return The username.
     */
    public String getUserName() {
        return userName;
    }

    /**
     * When this instance was created: milliseconds since epoch.
     * 
     * @return The epoch milliseconds since this instance was created.
     */
    public long getCreationMillis() {
        return creationMillis;
    }

    /**
     * When this instance will expire in milliseconds since epoch.
     * 
     * @return The epoch milliseconds of this request expiration.
     */
    public long getExpirationMillis() {
        return creationMillis + durationInSeconds*1000L;
    }

    /**
     * Get the duration in seconds of this MutedAlertRequest.
     * 
     * @return The duration in seconds.
     */
    public int getDurationInSeconds() {
        return durationInSeconds;
    }
    
    /**
     * Get the alertId regular expression for the Alerts to be muted by this
     * MutedAlertRequest instance.
     * 
     * @return The regular expression
     */
    public String getAlertIdRegEx() {
        return alertIdRegEx;
    }
    
    /**
     * Check if this MutedAlertRequest instance is still valid, i.e. is not
     * expired.
     * 
     * @return true if it's still valid, false otherwise.
     */
    public boolean isValid() {
        return System.currentTimeMillis() < getExpirationMillis();
    }
    
    /**
     * Check if the given alert for the given subsystem is muted.
     * 
     * @param subsystem
     * @param alert
     * @return true if the provided alert is muted.
     */
    public boolean isSubsystemAlertMuted(String subsystem, Alert alert) {
        return isMuted(subsystem,alert.getAlertId());
    }

    /**
     * Check if the given fully qualified alertId is muted.
     * A fully qualified alertId is a path like string whose first element is
     * the subsystem name.
     * 
     * @param alertId
     * @return true if the provided alertId is muted
     */
    public boolean isAlertIdMuted(String alertId) {
        if ( alertId.startsWith("/") ) {
            alertId = alertId.substring(1);
        }
        int slash = alertId.indexOf("/");
        if ( slash < 1 ) {
            throw new IllegalArgumentException("The provided alertId is not fully qualified. It must start with a subsystem name. "+alertId);
        }
        
        String subsystem = alertId.substring(0,slash);
        alertId = alertId.replace(subsystem+"/","");
     
        return isMuted(subsystem, alertId);
    }
    
    /**
     * Check if the given alertId matches the regular expression of this mute request.
     * This method does not check the request expiration time.
     * 
     * @param alertId Alert ID.
     * @return True if the regular expression associated with this mute request matches the specified alertId.
     */
    public boolean matches(String alertId) {
        return alertIdPattern.matcher(alertId).matches();
    }
    
    
    private boolean isMuted(String subsystem, String alertId) {
        return isValid() && subsystem.equals(subsystemName) && alertIdPattern.matcher(alertId).matches();        
    }

    @Override
    public String toString() {
        int secondsLeft = (int)((getExpirationMillis() - System.currentTimeMillis())/1000);
        return "Id: "+getUniqueId()+" subsystem: "+subsystemName+" regEx: "+alertIdRegEx+" muted by: "+userName+" validity left: "+secondsLeft+" (s)";
    }
    
}
