package org.lsst.ccs.bus.messages;

import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;

/**
 * Base class for a command reply.
 * Can be a CommandResult, CommandAck or CommandNack
 * This class provides an implementation for the encoding for all the subclasses.
 * If the content of the CommandReply is a Throwable it will be encoded to a
 * String containing the message and the stacktrace. If it is any other object,
 * the encoding will return the toString() method on that object.
 *
 * @author LSST CCS Team
 * @param <T> Template for the embedded reply object. It must be Serializable.
 */
public class CommandReply<T extends Serializable> extends CommandMessage<T,String> {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 65334356909L;

    /**
     * Main CommandReply constructor.
     * It is built from the corresponding CommandRequest and it encapsulates
     * the reply from its execution.
     * @param command The CommandRequest that originated the execution.
     * @param reply The result of the execution of the CommandRequest. If not successful
     *              the reply is a Throwable.
     */
    public CommandReply(CommandRequest command, T reply) {
        super(command.getOriginAgentInfo().getName(),command.getCorrelationId(),reply);
    }
        
    /**
     * Utility method to get the reply contained in this CommandReply.
     * It is equivalent to invoking the BusMessage getObject method.
     * 
     * @return The reply Object contained in this CommandReply.
     * @throws EmbeddedObjectDeserializationException if the embedded object cannot
     * be deserialized.
     */
    public Object getReply() {
        return getObject();
    }

    @Override
    public String getEncodedData() {
        return super.getEncodedData();
    }
    
    @Override
    protected String encodeObject(T obj) {
        if ( obj instanceof Throwable ) {
            Throwable t = (Throwable)obj;
            return t.getMessage();
        }
        return obj.toString();
    }
    

    
}
