package org.lsst.ccs.bus.data;

import static org.lsst.ccs.command.annotations.Command.NOT_DEFINED;

import java.io.Serializable;
import java.time.Instant;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.UUID;

import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

public class AgentLockInfo implements AgentLock, Serializable {

    private static final long serialVersionUID = 4416294268287507663L;
    private final String agentName;
    private final String owner;
    private final int maxLevel; // as associated to owner
    private final CCSTimeStamp timeStamp;
    private final String token;

    private final String originatingAgent; // for GUI informative purposes, and scripting consoles handling
    private boolean scriptingOriginator = false;
    private final String currentAgent; // for GUI informative purposes, current attaching console

    public enum Status {
        REQUESTED, // A lock is requested. Issue could be ack or rejected. Internal use, of interest to the LM only.
        ACKNOWLEDGED, // The lock manager has granted the lock
        ACK_LEGACY, // a fake ack for legacy subsystems not lock-manager aware
        RELEASED, // the lock has been released (unlock)
        REJECTED, // the lock request has been rejected by the lock manager
        LEVEL_UPDATE, // deprecated, to be removed, level updates are local
        REMINDER, // sent when a locked agent connects, to refresh the lock status after a restart
        ATTACH, // a console attached an existing lock
        DETACH, // a console detached an existing lock without releasing it
        INFO, // to notify connecting consoles of existing locks.
        LOGIN_INVISIBLE, // lock is now invisible because of login (held locks only)
        LOGIN_VISIBLE // lock is now visible because of login (held locks only)
    }

    private final Status status;

    // Factory methods, clearer than overloaded contructors

    // Creating an initial lock request — we don't know the max level yet
    public static AgentLockInfo createLockRequest(String agentName, String userId, AgentInfo originatingAgent) {
        return new AgentLockInfo(agentName, userId, 0, Status.REQUESTED, null, originatingAgent);
    }

    // Creating a completed lock request
    public static AgentLockInfo createLockRequest(AgentLockInfo request, int maxLevel) {
        return new AgentLockInfo(request.getAgentName(), request.getOwner(), maxLevel, Status.REQUESTED,
                request.getTimeStamp(), null, request.getOriginatingAgent(), request.isScriptingOriginator(),
                request.getCurrentAgent());
    }

    // initial attachment request
    public static AgentLockInfo createAttachRequest(String agentName, String userId, AgentInfo attachingAgent) {
        return new AgentLockInfo(agentName, userId, 0, Status.ATTACH, null, attachingAgent);
    }

    // completed attachment request
    public static AgentLockInfo createAttachRequest(AgentLockInfo attachRequest, AgentLockInfo originalLock) {
        return new AgentLockInfo(attachRequest.getAgentName(), attachRequest.getOwner(), originalLock.getMaxLevel(),
                Status.ATTACH, originalLock.getTimeStamp(), null, originalLock.getOriginatingAgent(),
                originalLock.isScriptingOriginator(), originalLock.getCurrentAgent());
    }

    public static AgentLockInfo createReject(AgentLockInfo request) {
        return new AgentLockInfo(request, Status.REJECTED, null);
    }

    public static AgentLockInfo createAcknowledge(AgentLockInfo request) {
        return new AgentLockInfo(request, Status.ACKNOWLEDGED, UUID.randomUUID().toString());
    }

    public static AgentLockInfo createAcknowledgeLegacy(AgentLockInfo request, int level) {
        return new AgentLockInfo(request, Status.ACK_LEGACY, level, UUID.randomUUID().toString());
    }

    public static AgentLockInfo createReminder(AgentLockInfo lock) {
        return new AgentLockInfo(lock, Status.REMINDER, lock.getToken());
    }

    public static AgentLockInfo createInfo(AgentLockInfo lock) {
        return new AgentLockInfo(lock, Status.INFO, lock.getToken());
    }

    public static AgentLockInfo createRelease(String agentName, String userId) {
        return new AgentLockInfo(agentName, userId, NOT_DEFINED, Status.RELEASED, null, null);
    }

    public static AgentLockInfo createDetach(String agentName, String userId, AgentInfo attachingAgent) {
        return new AgentLockInfo(agentName, userId, 0, Status.DETACH, null, attachingAgent);
    }

    public static AgentLockInfo createLoginVisble(AgentLockInfo lock) {
        return new AgentLockInfo(lock, Status.LOGIN_VISIBLE, lock.getToken());
    }

    public static AgentLockInfo createLoginInvisble(AgentLockInfo lock) {
        return new AgentLockInfo(lock, Status.LOGIN_INVISIBLE, lock.getToken());
    }



    private AgentLockInfo(String agentName, String owner, int maxLevel, Status status, CCSTimeStamp timeStamp,
            String token, AgentInfo originatingAgent) {
        this.agentName = agentName;
        this.owner = owner;
        this.maxLevel = maxLevel;
        this.status = status;
        this.timeStamp = timeStamp;
        this.token = token;
        this.originatingAgent = originatingAgent == null ? null : originatingAgent.getName();
        this.scriptingOriginator = originatingAgent == null ? false : originatingAgent.isScriptingConsole();
        this.currentAgent = originatingAgent == null ? null : originatingAgent.getName();
    }

    private AgentLockInfo(String agentName, String owner, int maxLevel, Status status, CCSTimeStamp timeStamp,
            String token, String originatingAgentName, boolean isScription, String currentAgentName) {
        this.agentName = agentName;
        this.owner = owner;
        this.maxLevel = maxLevel;
        this.status = status;
        this.timeStamp = timeStamp;
        this.token = token;
        this.originatingAgent = originatingAgentName;
        this.scriptingOriginator = isScription;
        this.currentAgent = originatingAgentName;
    }

    public AgentLockInfo(String agentName, String owner, int maxLevel, Status status, String token,
            AgentInfo originatingAgent) {
        this(agentName, owner, maxLevel, status, CCSTimeStamp.currentTime(), token, originatingAgent);
    }

    public AgentLockInfo(AgentLockInfo lock, Status status, String token) {
        this(lock.agentName, lock.owner, lock.maxLevel, status, CCSTimeStamp.currentTime(), token,
                lock.getOriginatingAgent(), lock.isScriptingOriginator(), lock.getCurrentAgent());
    }

    public AgentLockInfo(AgentLockInfo lock, Status status, int level, String token) {
        this(lock.agentName, lock.owner, level, status, CCSTimeStamp.currentTime(), token,
        lock.getOriginatingAgent(), lock.isScriptingOriginator(), lock.getCurrentAgent());
    }

    //
    @Override
    public String getAgentName() {
        return agentName;
    }

    @Override
    public String getOwner() {
        return owner;
    }

    public Status getStatus() {
        return status;
    }

    @Override
    public int getMaxLevel() {
        return maxLevel;
    }

    public CCSTimeStamp getTimeStamp() {
        return timeStamp;
    }

    @Override
    public String getToken() {
        return token;
    }

    public String getOriginatingAgent() {
        return originatingAgent;
    }

    public boolean isScriptingOriginator() {
        return scriptingOriginator;
    }

    public String getCurrentAgent() {
        return currentAgent;
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 67 * hash + Objects.hashCode(this.agentName);
        hash = 67 * hash + Objects.hashCode(this.owner);
        hash = 67 * hash + this.maxLevel;
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final AgentLockInfo other = (AgentLockInfo) obj;
        if (!Objects.equals(this.agentName, other.agentName)) {
            return false;
        }
        if (!Objects.equals(this.owner, other.owner)) {
            return false;
        }
        if (!Objects.equals(this.token, other.token)) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return "Lock " + agentName + " by " + owner + " " + status + " at " + timeStamp.getUTCInstant().toString()
                + " [" + token + "]";
    }
    
    static public String asString(AgentLock lock) {
        if (lock == null) return "null";
        StringBuilder sb = new StringBuilder();
        sb.append("[Agent: ").append(lock.getAgentName()).append(", owner: ").append(lock.getOwner());
        try {
            AgentLockInfo theLock = (AgentLockInfo) lock;
            sb.append(", ").append(theLock.getStatus());
            sb.append(", current: ").append(theLock.getCurrentAgent()).append(", origin: ").append(theLock.getOriginatingAgent());
        } catch (ClassCastException x) {
            sb.append(", class: ").append(lock.getClass().getName());
        }
        return sb.append("]").toString();
    }

    /**
     * Class that allows the serialization of the AgentLockInfo
     *
     * We use the PersistenceService to make sure the locks are stored at any given
     * time. The PersistenceService needs the persisted object to be convertible to
     * and from a String, which the AgentLockInfo implementation is not.
     *
     * This class alleviates that issue.
     *
     * @author Alexandre Boucaud
     */
    public static class AgentLockInfoString {

        private AgentLockInfo myLockInfo;

        static String delimiter = ";";

        public AgentLockInfoString(AgentLockInfo info) {
            this.myLockInfo = info;
        }

        public AgentLockInfoString(String stringArgs) {
            String[] args = stringArgs.split(delimiter);

            String agentName = args[0];
            String owner = args[1];
            int maxLevel = Integer.parseInt(args[2]);
            Status status = Status.valueOf(args[3]);
            CCSTimeStamp timeStamp = new CCSTimeStamp(Instant.parse(args[4]), Instant.parse(args[5]));
            String token = args[6];
            String originatingConsole = args[7];
            boolean scriptingOriginator = args[8].equals("T");
            String currentConsole = args[9];
            this.myLockInfo = new AgentLockInfo(agentName, owner, maxLevel, status, timeStamp, token,
                    originatingConsole, scriptingOriginator, currentConsole);
        }

        public AgentLockInfo getLockInfo() {
            return myLockInfo;
        }

        @Override
        public String toString() {
            StringJoiner lockInfo = new StringJoiner(delimiter);

            lockInfo.add(myLockInfo.getAgentName());
            lockInfo.add(myLockInfo.getOwner());
            lockInfo.add(String.valueOf(myLockInfo.getMaxLevel()));
            lockInfo.add(myLockInfo.getStatus().name());
            lockInfo.add(myLockInfo.getTimeStamp().getTAIInstant().toString());
            lockInfo.add(myLockInfo.getTimeStamp().getUTCInstant().toString());
            lockInfo.add(myLockInfo.getToken());
            lockInfo.add(myLockInfo.getOriginatingAgent());
            lockInfo.add(myLockInfo.isScriptingOriginator() ? "T" : "F");
            lockInfo.add(myLockInfo.getCurrentAgent());

            return lockInfo.toString();
        }
    }

}
