package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A Class containing the current configuration data.
 * 
 * @author The LSST CCS Team
 */
public class ConfigurationData implements Serializable {

    private static final long serialVersionUID = 5968743302322344L;
    
    private ConfigurationState configurationState;    
    private Map<String,String> configurationData = new HashMap<>();
    //Map of configuration parameter values
    //This is used to know if a configuration parameter is dirty
    //and what its configured value is. This map contains a value
    //only if the parameter is dirty.
    private Map<String,String> configuredData = new HashMap<>();
    private Map<String,String> categoryDescription = new HashMap<>();
    private String configurationDescription;
    private String fullConfigurationDescription;
    private Map<String,Long> categoryChecksum = new HashMap<>();
    private String configOperation;
    private CCSTimeStamp ccsTimestamp;
    private final List<String> recentChanges = new ArrayList<>();
    
    /**
     * Get the full configuration description.
     * @return The full configuration description.
     */
    public String getConfigurationDescription() {
        return configurationDescription;
    }
    public String getFullConfigurationDescription() {
        return fullConfigurationDescription;
    }
    
    /**
     * Get the full category description for the given category.
     * @param category The provided category name.
     * @return  The category description for the provided category name.
     */
    public String getCategoryDescription(String category) {
        return categoryDescription.get(category);
    }

    
    /**
     * Get the current value for a given configuration parameter.
     * 
     * @param parameterPath The configuration parameter path.
     * @return The configuration parameter value.
     */
    public String getConfigurationParameterValue(String parameterPath) {
        return configurationData.get(parameterPath);
    }

    
    /**
     * Get the current ConfigurationState
     * @return The ConfigurationState
     */
    public ConfigurationState getConfigurationState() {
        return configurationState;
    }
    
    /**
     * Get the data values checksum for the given category.
     * @param category The input category.
     * @return The checksum for the provided category.
     */
    public long getCategoryChecksum(String category) {
        return categoryChecksum.get(category);
    }   
    
    /**
     * Get the configuration operation that lead to this publication.
     * @return the configuration operation.
     */
    public String getConfigurationOperation() {
        return configOperation;
    }
    
    public Map<String,String> getCurrentValues() {
        return configurationData;
    }
    
    public Map<String,String> getConfiguredValues() {
        return configuredData;
    }
    
    public Set<String> getCategories() {
        return categoryDescription.keySet();
    }
    
    public CCSTimeStamp getCCSTimeStamp() {
        return ccsTimestamp;
    }
    
    public List<String> getLatestChanges() {
        return recentChanges;
    }
    
    public static ConfigurationData buildConfigurationData(ConfigurationInfo ci, List<ConfigurationParameterInfo> pars, String co) {
        
        ConfigurationData configData = new ConfigurationData();

        configData.configurationState = ci.getConfigurationState();
        configData.configurationDescription = ci.getConfigurationDescription();
        configData.fullConfigurationDescription = ci.getFullConfigurationDescription();
        configData.configOperation = co;

        for ( String category : ci.getCategorySet() ) {
            configData.categoryDescription.put(category,ci.getConfigurationDescriptionObject().getCategoryTag(category).toString());
            configData.categoryChecksum.put(category,ci.getCategoryDataChecksum(category));
        }

        for (ConfigurationParameterInfo par : pars ) {
            String parPath = par.getPathName();
            String currentValue = par.getCurrentValue();
            configData.configurationData.put(parPath, currentValue);
            String configuredValue = par.getConfiguredValue();
            if ( ! currentValue.equals(configuredValue) ) {
                configData.configuredData.put(parPath, configuredValue);
            }
        }

        configData.ccsTimestamp = ci.getCCSTimeStamp();
        configData.recentChanges.clear();
        for ( ConfigurationParameterInfo cpi : ci.getLatestChanges()) {
            configData.recentChanges.add(cpi.getPathName());            
        }
        
        
        return configData;
    }
    
    
}
