package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.data.DataProviderInfo.Attribute;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;


/**
 * Data dictionary implementation that minimizes memory footprint and serialized size at the expense of access speed.
 * Instances of this class are immutable.
 * 
 * <h4>Implementation notes</h4>
 * Currently, extra complexity is added by the fact that we associate key as well as path with every data channel.
 * This is left over from supporting both full and short paths, and is expected to be removed in later versions.
 * For now, {@code int keyFlag} is associated with every tree node to tell how to extract path and key from the full path:
 * <ul>
 * <li>keyFlag = NO_PATH : not a terminal node
 * <li>keyFlag = NULL_PATH : path = null, key = fullPath
 * <li>keyFlag = EMPTY_PATH : path = "", key = fullPath
 * <li>keyFlag less than 0 : abs(keyFlag) is the number of segments in key, fullPath = path + key
 * <li>others : keyFlag is the number of segments in key, fullPath ends with key
 * </ul>
 *
 * @author onoprien
 */
public final class DataDictionary implements DataProviderDictionary {

// -- Fields : -----------------------------------------------------------------
    
    static private final int NO_PATH = 1000;
    static private final int NULL_PATH = 1001;
    static private final int EMPTY_PATH = 1002;
    static private final int NULL_KEY = 1003;
    static private final int EMPTY_KEY = 1004;
    
    static private final long serialVersionUID = 1L;
    
    private final Node root;
    private final int size;

// -- Life cycle : -------------------------------------------------------------
    
    public DataDictionary(DataProviderDictionary other) {
        this(other.getDataProviderInfos());
    }
    
    public DataDictionary(List<DataProviderInfo> data) {
        
        // Build full (not optimized) tree
        
        int size = 0;
        BuilderNode builderRoot = new BuilderNode(null, Collections.emptyMap());
        for (DataProviderInfo d : data) {
            String[] path = d.getFullPath().split("/");
            Map<Attribute,String> att = new HashMap<>();
            for (Attribute a : d.getAttributes()) {
                att.put(a, d.getAttributeValue(a));
            }
            BuilderNode  parent = builderRoot;
            for (int i = 0; i < path.length-1; i++) {
                String name = path[i];
                BuilderNode  child = parent.children.get(name);
                if (child == null) {
                    child = new BuilderNode(name, att);
                    parent.children.put(name, child);
                    att.clear();
                } else {
                    Iterator<Map.Entry<Attribute,String>> it = child.att.entrySet().iterator();
                    while (it.hasNext()) {
                        Map.Entry<Attribute,String> e = it.next();
                        Attribute key = e.getKey();
                        String oldValue = e.getValue();
                        String newValue = att.get(key);
                        if (Objects.equals(oldValue, newValue)) {
                            att.remove(key);
                        } else {
                            it.remove();
                            child.children.values().forEach(node -> {
                                node.att.put(key, oldValue);
                            });
                            child.leaves.values().forEach(node -> {
                                node.att.put(key, oldValue);
                            });
                        }
                    }
                }
                parent = child;
            }
            String lastName = path[path.length-1];
            BuilderNode leaf = new BuilderNode(lastName, att);
            if (parent.leaves.put(lastName, leaf) != null) {
                throw new RuntimeException("Duplicate path: "+ d.getFullPath());
            }            
            size++;
            String p = d.getPath();
            String k = d.getKey();
            if (p == null) {
                leaf.keyFlag = NULL_PATH;
            } else if (p.isEmpty()) {
                leaf.keyFlag = EMPTY_PATH;
            } else if (k == null) {
                leaf.keyFlag = NULL_KEY;
            } else if (k.isEmpty()) {
                leaf.keyFlag = EMPTY_KEY;
            } else if (p.endsWith(k)) {
                leaf.keyFlag = k.split("/").length;
            } else {
                leaf.keyFlag = - k.split("/").length;
            }
        }
        
        // Merge identical branches
        
        tryToMergeChildren(builderRoot);
        
        // Convert to final tree
        
        Map<Map<Attribute,String>, Map.Entry<Attribute[],String[]>> maps = new HashMap<>(2048);
        root = convert(builderRoot, maps);
        this.size = size;
    }
    
    /**
     * Try to merge node's children.
     * Also compute hash for this node to speed up future comparisons.
     */
    private void tryToMergeChildren(BuilderNode node) {
        node.hash = node.att.hashCode();
        if (node.children.isEmpty() && node.leaves.isEmpty()) {
            return;
        }
        ArrayList<Map<String,BuilderNode>> all = new ArrayList<>(2);
        all.add(node.children);
        all.add(node.leaves);
        for (Map<String,BuilderNode> daughters : all) {
            daughters.values().forEach(child -> {
                tryToMergeChildren(child);
                node.hash = (97 * node.hash) + child.hash;
            });
            LinkedList<BuilderNode> nodes = new LinkedList<>(daughters.values());
            while (!nodes.isEmpty()) {
                BuilderNode child1 = nodes.pollFirst();
                Iterator<BuilderNode> it = nodes.iterator();
                while (it.hasNext()) {
                    BuilderNode child2 = it.next();
                    if (child1.equals(child2)) {
                        child1.names.addAll(child2.names);
                        it.remove();
                        child2.names.forEach(name -> {
                            daughters.remove(name);
                        });
                    }
                }
            }
        }
    }
    
    private Node convert(BuilderNode builderNode, Map<Map<Attribute,String>, Map.Entry<Attribute[],String[]>> maps) {
        
        String names = String.join("+", builderNode.names);
        
        Map.Entry<Attribute[], String[]> att;
        if (builderNode.att.isEmpty()) {
            att = new AbstractMap.SimpleImmutableEntry<>(null, null);
        } else {
            att = maps.get(builderNode.att);
            if (att == null) {
                int n = builderNode.att.size();
                Attribute[] keys = new Attribute[n];
                String[] values = new String[n];
                int i=0;
                for (Map.Entry<Attribute, String> e : builderNode.att.entrySet()) {
                    keys[i] = e.getKey();
                    values[i++] = e.getValue();
                }
                att = new AbstractMap.SimpleImmutableEntry<>(keys, values);
                maps.put(builderNode.att, att);
            }
        }
        
        Node[] children;
        if (builderNode.children.isEmpty() && builderNode.leaves.isEmpty()) {
            children = null;
        } else {
            children = new Node[builderNode.children.size() + builderNode.leaves.size()];
            int i = 0;
            for (BuilderNode child : builderNode.children.values()) {
                children[i++] = convert(child, maps);
            }
            for (BuilderNode child : builderNode.leaves.values()) {
                children[i++] = convert(child, maps);
            }
        }
        
        return new Node(names, att.getKey(), att.getValue(), children, builderNode.keyFlag);
    }


// -- Implementing DataProviderDictionary : ------------------------------------

    @Override
    public List<DataProviderInfo> getDataProviderInfos() {
        return new AbstractSequentialList<DataProviderInfo>() {
            @Override
            public ListIterator<DataProviderInfo> listIterator(int index) {
                if (index < 0 || index > size) {
                    throw new IndexOutOfBoundsException();
                }
                ListIterator<DataProviderInfo> it = new Iter();
                while (index-- > 0) {
                    it.next();
                }
                return it;
            }
            @Override
            public int size() {
                return size;
            }            
        };
    }
    
    @Override
    public DataProviderInfo getDataProviderInfoForPath(String path) {
        Map<Attribute,String> att = new LinkedHashMap<>();
        String[] ss = path.split("/");
        Node parent = root;
        int n = ss.length;
        for (String name : ss) {
            Node child = null;
            if (--n > 0) {
                for (Node node : parent.children) {
                    if (node.containsName(name)) {
                        child = node;
                        break;
                    }
                }
            } else { // last segment, leaves first
                for (int i=parent.children.length; i>0; ) {
                    Node node = parent.children[--i];
                    if (node.containsName(name)) {
                        child = node;
                        break;
                    }
                }
            }
            if (child == null) return null;
            if (child.attKeys != null) {
                for (int i=0; i < child.attKeys.length;  i++) {
                    att.put(child.attKeys[i], child.attValues[i]);
                }
            }
            parent = child;
        }
        return makeDPI(path, parent.keyFlag, att);
    }

    @Deprecated
    @Override
    public Set<String> getGroups() {
        return Collections.singleton("");
    }

    @Deprecated
    @Override
    public List<DataProviderInfo> getDataProviderDescriptionsForGroup(String group) {
        return group == null || !group.isEmpty() ? Collections.emptyList() : getDataProviderInfos();
    }

    @Deprecated
    @Override
    public CCSTimeStamp getCCSTimeStamp() {
        return CCSTimeStamp.currentTime();
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /**
     * Constructs a new instance.
     * This method can be replaced by simple constructor once we stop using keys.
     */
    static private DataProviderInfo makeDPI(String fullPath, int keyFlag, Map<Attribute,String> attributes) {
        switch (keyFlag) {
            case NULL_PATH:
                return new DataProviderInfo(null, fullPath, attributes);
            case EMPTY_PATH:
                return new DataProviderInfo("", fullPath, attributes);
            case NULL_KEY:
                return new DataProviderInfo(fullPath, null, attributes);
            case EMPTY_KEY:
                return new DataProviderInfo(fullPath, "", attributes);
            case NO_PATH:
                return null;
            default:
                List<String> ss = Arrays.asList(fullPath.split("/"));
                int n = ss.size();
                if (keyFlag == n) {
                    return new DataProviderInfo(fullPath, fullPath, attributes);
                } else {
                    if (keyFlag > 0) {
                        String key = String.join("/", ss.subList(n - keyFlag, n));
                        return new DataProviderInfo(fullPath, key, attributes);
                    } else {
                        String path = String.join("/", ss.subList(0, n + keyFlag));
                        String key = String.join("/", ss.subList(n + keyFlag, n));
                        return new DataProviderInfo(path, key, attributes);
                    }
                }
        }

    }
    
    
// -- Local classes : ----------------------------------------------------------
    
     static private class BuilderNode {
         
        final SortedSet<String> names;
        final Map<Attribute,String> att;
        final Map<String,BuilderNode> children;
        final Map<String,BuilderNode> leaves;
        int keyFlag = NO_PATH;
        int hash;
         
        BuilderNode(String name, Map<Attribute,String> att) {
            names = new TreeSet<>();
            if (name != null) names.add(name);
            this.att = new TreeMap<>(att);
            children = new TreeMap<>();
            leaves = new TreeMap<>();
        }         

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof BuilderNode)) return false;
            BuilderNode other = (BuilderNode) obj;
            if (hash != other.hash) return false;
            if (keyFlag != other.keyFlag) return false;
            if (!Objects.equals(att, other.att)) return false;
            if (!Objects.equals(children, other.children)) return false;
            return Objects.equals(leaves, other.leaves);
        }

        @Override
        public int hashCode() {
            int h = 7;
            h = 73 * h + Objects.hashCode(att);
            h = 73 * h + Objects.hashCode(children);
            h = 73 * h + Objects.hashCode(leaves);
            h = 73 * h + keyFlag;
            h = 73 * h + hash;
            return h;
        }
     }
     
     static private class Node implements Serializable {
         
         final String names; // "name1+name2+...+nameN
         final Attribute[] attKeys; // null if no attributes
         final String[] attValues; // null if no attributes
         final Node[] children; // null if no children
         final int keyFlag;
         
         Node(String names, Attribute[] attKeys, String[] attValues, Node[] children, int keyFlag) {
             this.names = names;
             this.attKeys = attKeys;
             this.attValues = attValues;
             this.children = children;
             this.keyFlag = keyFlag;
         }
         
         boolean containsName(String name) {
             if (names.equals(name)) return true;
             return Arrays.binarySearch(names.split("\\+"), name) >= 0;
         }
         
     }
     
     private class Iter implements ListIterator<DataProviderInfo> {
         
         private int nextIndex;
         private final IterNode[] nodes = new IterNode[50];
         private int itNodeIndex;
         
         Iter() {
             nextIndex = 0;
             IterNode itRoot = new IterNode();
             itRoot.node = root;
             itRoot.paths = null;
             itRoot.att = Collections.emptyMap();
             itRoot.pathIndex = -1;
             itRoot.childIndex = 0;
             nodes[0] = itRoot;
             itNodeIndex = 0;
             seek(true);
         }

         @Override
         public boolean hasNext() {
             return nextIndex < size;
         }

         @Override
         public DataProviderInfo next() {
             if (nextIndex == size) throw new NoSuchElementException("No next element");
             IterNode itNode = nodes[itNodeIndex];
             DataProviderInfo out = makeDPI(itNode.paths.get(itNode.pathIndex), itNode.node.keyFlag, itNode.att);
             if (++nextIndex < size) stepForward();
             return out;
         }

         @Override
         public boolean hasPrevious() {
             return nextIndex > 0;
         }

         @Override
         public DataProviderInfo previous() {
             if (nextIndex == 0) throw new NoSuchElementException("No previous element");
             if (nextIndex-- < size) stepBack();
             IterNode itNode = nodes[itNodeIndex];
             return makeDPI(itNode.paths.get(itNode.pathIndex), itNode.node.keyFlag, itNode.att);
         }

         @Override
         public int nextIndex() {
             return nextIndex;
         }

         @Override
         public int previousIndex() {
             return nextIndex - 1;
         }

         @Override
         public void remove() {
             throw new UnsupportedOperationException("Not supported by unmodifiable list.");
         }

         @Override
         public void set(DataProviderInfo e) {
             throw new UnsupportedOperationException("Not supported by unmodifiable list.");
         }

         @Override
         public void add(DataProviderInfo e) {
             throw new UnsupportedOperationException("Not supported by unmodifiable list.");
         }
         
         private void stepBack() {
             IterNode itNode = nodes[itNodeIndex];
             if (itNode.pathIndex-- == 0) {
                 while (true) {
                     itNode = nodes[--itNodeIndex];
                     if (itNode.childIndex > 0) {
                         itNode.childIndex--;
                         seek(false);
                         return;
                     }
                 }
             }
         }
         
         private void stepForward() {
             IterNode itNode = nodes[itNodeIndex];
             if (++itNode.pathIndex == itNode.paths.size()) {
                 while (true) {
                     itNode = nodes[--itNodeIndex];
                     if (itNode.childIndex < itNode.node.children.length - 1) {
                         itNode.childIndex++;
                         seek(true);
                         return;
                     }
                 }
             }
         }
         
         private void seek(boolean forward) {
             IterNode itParent = nodes[itNodeIndex];
             Node child = itParent.node.children[itParent.childIndex];
             IterNode itChild = new IterNode();
             nodes[++itNodeIndex] = itChild;
             itChild.node = child;
             String[] names = child.names.split("\\+");
             if (itParent.paths == null) {
                 itChild.paths = new ArrayList<>(Arrays.asList(names));
             } else {
                 itChild.paths = new ArrayList<>(itParent.paths.size()*names.length);
                 for (String path : itParent.paths) {
                     for (String name : names) {
                         itChild.paths.add(path +"/"+ name);
                     }
                 }
             }
             itChild.att = new LinkedHashMap<>(Attribute.values().length*2);
             itChild.att.putAll(itParent.att);
             if (child.attKeys != null) {
                 for (int i=0; i < child.attKeys.length; i++) {
                     itChild.att.put(child.attKeys[i], child.attValues[i]);
                 }
             }
             if (child.keyFlag == NO_PATH) {
                 itChild.pathIndex = -1;
                 itChild.childIndex = forward ? 0 : child.children.length-1;
                 seek(forward);
             } else {
                 itChild.pathIndex = forward ? 0 : itChild.paths.size()-1;
                 itChild.childIndex = -1;
             }
             
         }
         
     }
     
     static private class IterNode {
         Node node;
         ArrayList<String> paths;
         Map<Attribute,String> att;
         int childIndex; // index of the  child being explored; -1 before
         int pathIndex; // path to be returned by next(), -1 if not terminal node, paths.size() if all paths have been used
     }
    
    
// - Testing : -----------------------------------------------------------------
    
    static public void main(String... args) {
        
        
        System.out.println("a".matches("[-\\w]"));
        
        Pattern p = Pattern.compile("([-.\\w]+)(/[-.\\w]+)+");
        Matcher m = p.matcher("ag.ent/ds.sf/gf-grr/k_l.");
        if (m.matches()) {
            System.out.println("Agent = "+ m.group(1));
        } else {
            System.out.println("No match.");
        }
        
        
    }
        

    
}
