package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.time.Duration;
import java.util.Objects;
import java.util.Properties;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A class to encapsulate an agent's name and type.
 * This class is used to represent an Agent across the buses.
 * The Agent's name is its unique name on the Buses.
 * Its type defines its role on the buses.
 * 
 * Two AgentInfo instances are considered equal if they have the same name and type.
 * When two AgentInfo instances are equal, they represent the same Agent on
 * the buses.
 *
 * @author emarin
 */
//REVIEW: Since the name is guaranteed to be unique, it seems not necessary
//to test the type for equality.
public final class AgentInfo implements Serializable {
    
    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 716901596341446997L;

    private final String name;
    private final AgentType type;
    private final Properties agentProps;

    public static final String AGENT_NAME_PROP = "agentName";
    public static final String AGENT_TYPE_PROP = "agentType";
    public static final String AGENT_OFFLINE_PROP = "agentOffline";

    public static final String AGENT_JXM_PORT_PROP = "org.lsst.ccs.agent.jmxport";
    
    //The time the Agent was started, which is roughly the time this
    //object was created.
    private final CCSTimeStamp agentStartTime;
    private CCSTimeStamp agentOperationalTime;
    private CCSTimeStamp agentJoinedTheBusesTime;
    
    /**
     * The base constructor of an AgentInfo. It takes the Agent's name and its 
     * AgentType.
     * 
     * @param name The name of the Agent as defined on the Buses.
     * @param type The AgentType of the Agent. This determines its role.
     */
    public AgentInfo(String name, AgentType type) {
        this(name, type, new Properties());
    }

    /**
     * The base constructor of an AgentInfo. It takes the Agent's name and its 
     * AgentType.
     * 
     * @param name The name of the Agent as defined on the Buses.
     * @param type The AgentType of the Agent. This determines its role.
     * @param agentProps A set of properties for this agent.
     */
    public AgentInfo(String name, AgentType type, Properties agentProps) {
        this.agentProps = agentProps;
        this.name = name;
        this.type = type;
        
        this.agentProps.setProperty(AGENT_NAME_PROP, name);
        this.agentProps.setProperty(AGENT_TYPE_PROP, type.name());
        this.agentStartTime = CCSTimeStamp.currentTime();
    }

    /**
     * The Agent's name on the Buses. This must be unique.
     * 
     * @return The name of the Agent on the Buses.
     */
    public String getName() {
        return name;
    }
    
    /**
     * The Agent's AgentType. This represents its role on the buses.
     * 
     * @return The AgentType of the Agent.
     */
    public AgentType getType() {
        return type;
    }

    /**
     * Get the value of the property for this Agent.
     * null is returned if the given property is not defined.
     * 
     * @param property The name of the property to retrieve.
     * @return         The value of the Agent property. null if the given 
     *                 property is not set.
     */
    public String getAgentProperty(String property) {
        return getAgentProperty(property, null);
    }
    
    /**
     * Get the value of the property for this Agent.
     * null is returned if the given property is not defined.
     * 
     * @param property The name of the property to retrieve.
     * @param def      The default value to return.
     * @return         The value of the Agent property. The value of the def parameter if the given 
     *                 property is not set.
     */
    public String getAgentProperty(String property, String def) {
        return agentProps != null ? agentProps.getProperty(property, def) : def;
    }

    /**
     * Check if a given property is set for this Agent.
     * 
     * @param property The name of the property to check.
     * @return         true if the property is set for this Agent. false otherwise.
     */
    public boolean hasAgentProperty(String property) {
        return getAgentProperty(property) != null;
    }
    
    /**
     * Get the Properties for this AgentInfo.
     * This method is temporary and should not be used to change Agent Properties.
     * It should only be used by the AgentPropertiesService.
     * 
     * @return the AgentInfo Properties object.
     * 
     */
    public Properties getAgentProperties() {
        return agentProps;
    }
    
    /**
     * Get the CCSTimeStamp corresponding to the time this Agent
     * was started.
     * NOTE: this timestamp can be null for AgentInfo objects sent across the
     * buses that still don't have this field defined. Client code using this
     * method should protect against null until all the agents on the cluster
     * have been updated to a newer version of the toolkit.
     * @return 
     */
    public CCSTimeStamp getAgentStartTime() {
        return agentStartTime;
    }
    
    /** 
     * Public method to set the time the Agent became OPERATIONAL.
     * This method is meant to be used by the Agent class right before going into
     * OPERATIONAL state.
     * 
     */
    public final synchronized void setAgentOperationalTime() {
        if ( agentOperationalTime != null ) {
            throw new RuntimeException("The Agent Operational time can be set only once! Something went wrong!");
        }
        agentOperationalTime = CCSTimeStamp.currentTime();
    }
    
    /**
     * Get the time the Agent went operational.
     * For compatibility with older versions if the operational time has not
     * been set we return the startTime padded by 3 minutes.
     */
    public synchronized final CCSTimeStamp getAgentOperationalTime() {
        if ( agentOperationalTime == null ) {
            agentOperationalTime = CCSTimeStamp.currentTimeFromMillis(agentStartTime.getUTCInstant().plus(Duration.ofMinutes(3)).toEpochMilli());
        }
        return agentOperationalTime; 
    }

    /** 
     * Public method to set the time the Agent became Joined The Buses.
     * This method is meant to be used by the Agent class right after the
     * Agent has joined the buses.
     * 
     */
    public final synchronized void setAgentJoinedTheBusesTime() {
        if ( agentJoinedTheBusesTime != null ) {
            throw new RuntimeException("The Agent Operational time can be set only once! Something went wrong!");
        }
        agentJoinedTheBusesTime = CCSTimeStamp.currentTime();
    }
    
    /**
     * Get the time the Agent joined the buses.
     * For compatibility with older versions if the time it joined the buses has not
     * been set we return the Operational Time.
     */
    public synchronized final CCSTimeStamp getAgentJoinedTheBusesTime() {
        if ( agentJoinedTheBusesTime == null ) {
            agentJoinedTheBusesTime = getAgentOperationalTime();
        }
        return agentJoinedTheBusesTime; 
    }

    
    /**
     * Enumeration of known agent types.
     * 
     */
    public static enum AgentType {

        LISTENER("Listener"),
        CONSOLE("Console"),
        WORKER("Worker"),
        SERVICE("Service"),
        MCM("MCM"),
        OCS_BRIDGE("OCS bridge"),
        LOCK_MANAGER("Lock manager"),
        MMM("MMM");

        private final String displayName;

        private AgentType(String displayName) {
            this.displayName = displayName;
        }

        /**
         * Get the name of the AgentType for display purposes.
         * 
         * @return The display name of the AgentType.
         */
        //REVIEW: this method seems to be for GUI display purposes.
        //Maybe it does not belong here.
        public String displayName() {
            return displayName;
        }
        
    }

    /**
     * Utility method to decide this AgentInfo belongs to a WORKER or a Service
     * agent
     *
     * @return true if the type is equal or above WORKER.
     */
    public boolean isAgentWorkerOrService() {
        return type.compareTo(AgentType.WORKER) >= 0;
    }
    
    @Override
    public String toString() {
        return name + ":" + type;
    }

    @Override
    public boolean equals(Object a) {
        if (!(a instanceof AgentInfo)) {
            return false;
        } else {
            AgentInfo agent = (AgentInfo) a;
            return getName().equals(agent.getName()) && getType().equals(agent.getType());
        }

    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 89 * hash + Objects.hashCode(this.name);
        hash = 89 * hash + Objects.hashCode(this.type);
        return hash;
    }
    
    /**
     * Return true if this is a scripting console.
     */
    public boolean isScriptingConsole() {
        if ( getType().equals(AgentType.CONSOLE) ) {
            //We are using property org.lsst.ccs.console.type to distinguish
            //between graphical/scripting/shell consoles. 
            //See https://jira.slac.stanford.edu/browse/LSSTCCS-1874
            //If this property is not defined we assume the console is a graphical
            //console, i.e. we adopt the most conservative option
            String consoleType = getAgentProperty("org.lsst.ccs.console.type");
            if ( consoleType != null && ! consoleType.isEmpty() ) {
                return "scripting".equals(consoleType);
            }
            return false;
        }
        return false;
    }
    
    /**
     * Return true if this is a graphical console;
     */
    public boolean isGraphicalConsole() {
        if ( getType().equals(AgentType.CONSOLE) ) {
            //We are using property org.lsst.ccs.console.type to distinguish
            //between graphical/scripting/shell consoles. 
            //See https://jira.slac.stanford.edu/browse/LSSTCCS-1874
            //If this property is not defined we assume the console is a graphical
            //console, i.e. we adopt the most conservative option
            String consoleType = getAgentProperty("org.lsst.ccs.console.type");
            if ( consoleType != null && ! consoleType.isEmpty() ) {
                if ( "scripting".equals(consoleType) || "shell".equals(consoleType) ) {
                    return false;
                } else {
                    return true;
                }
            }
            return true;
        }
        return false;
    }
    
}
