package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.bootstrap.SubstitutionTokenUtils;

/**
 * A class describing a single category tag.
 * A single category tag is described by:
 *  - the source of the data
 *  - the category of the configuration parameters it describes
 *  - the tag
 *  - the version
 * 
 * A SingleCategoryTag is expressed as: category:source/tag(version)
 * 
 * 
 * @author The LSST CCS Team
 */
public class SingleCategoryTag implements Serializable {
         
    private static final long serialVersionUID = 8606602350889253605L;
 
    private final String category, source, tag, version, resolvedVersion;
    private final String defaultSource;
    
    public final static String DEFAULT_VERSION = "d";
    public final static String UNSPECIFIED_VERSION = "";
    
    public SingleCategoryTag(String source, String tag, String category) {
        this(source, tag, category, DEFAULT_VERSION, DEFAULT_VERSION);
    }
    
    public SingleCategoryTag(String source, String tag, String category, String requestedVersion) {
        this(source,tag,category,requestedVersion,requestedVersion);
    }
    
    public SingleCategoryTag(String source, String tag, String category, String requestedVersion, String resolvedVersion) {
        this.category = category;
        this.source = source;
        this.tag = tag;
        if ( category == null ) {
            throw new RuntimeException("Cannot build a SingleCategory object with a null category!!");
        }
        if ( tag == null ) {
            throw new RuntimeException("Cannot build a SingleCategory object with a null tag!!");
        }
        if ( tag.isEmpty() ) {
            throw new IllegalArgumentException("Category tag cannot be empty");
        }
        this.version = requestedVersion == null ? UNSPECIFIED_VERSION : requestedVersion;
        this.resolvedVersion = resolvedVersion == null || resolvedVersion.isEmpty() ? DEFAULT_VERSION : resolvedVersion;
        
        this.defaultSource = getDefaultSourceForCategory(category);
    }

    //Copy constructor
    public SingleCategoryTag(SingleCategoryTag t) {
        this(t, t.resolvedVersion);
    }
    
    //Copy constructor with updated version
    public SingleCategoryTag(SingleCategoryTag t, String version) {
        this(t.source, t.tag, t.category, t.version, version);
    }
    
    public String getCategory() {
        return category;
    }

    public String getSource() {
        return source;
    }

    public String getTag() {
        return tag;
    }
    
    public String getInputVersion() {
        return version;
    }

    public String getRequestedVersion() {
        return version.equals(UNSPECIFIED_VERSION) ? DEFAULT_VERSION : version;
    }
    
    public String getResolvedVersion() {
        return resolvedVersion;
    }

    public String getVersion() {
        if ( resolvedVersion != null && ! resolvedVersion.equals(version) )
            return resolvedVersion;
        return version;
    }
    
    public boolean isRequestedVersionAnInteger() {
        try {
            Integer.parseInt(getRequestedVersion());
            return true;
        } catch (NumberFormatException nfe) {
            return false;
        }
    }
    
    @Override
    public String toString() {
        return convertToString(true,false);
    }

    public String convertToString(boolean showCategory, boolean showDefaultSource) {
        StringBuilder sb = new StringBuilder();
        if ( showCategory ) {
            sb.append(category).append(":");
        }
        if ( showDefaultSource || !defaultSource.equals(source) ) {
            String sourceStr = (source != null && ! source.isEmpty()) ? source+"/" : "";
            sb.append(sourceStr);
        }
        sb.append(tag).append("(").append(getRequestedVersion());
        if ( resolvedVersion != null && !resolvedVersion.equals(getRequestedVersion()) ) {
            sb.append("=").append(resolvedVersion);
        }
        sb.append(")");
        return sb.toString();
    }

    @Override
    public boolean equals(Object obj) {
        if ( obj instanceof SingleCategoryTag ) {
            SingleCategoryTag t = (SingleCategoryTag)obj;
            if ( t.tag.equals(tag) && t.category.equals(category) ) {
                if ( source == null ) {
                    return t.source == null;
                }
                return source.equals(t.source);
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return tag.hashCode() + (source != null ? 23*source.hashCode() : 0 ) + 154 * category.hashCode();
    }
    
    private static Map<String,String> defaultCategorySources = new ConcurrentHashMap<>();
    private static String defaultForAllCategories = "<description>";
    
    public static void resetDefaultSources() {
        defaultForAllCategories = "<description>";
        defaultCategorySources.clear();
    }
    
    public static void setDefaultForAllCategories(String defaultForAllCats) {
        defaultForAllCategories = defaultForAllCats;
    }
    public static void setDefaultForCategory(String category, String defaultForCat) {
        defaultCategorySources.put(category, defaultForCat);        
    }
    
    public static boolean hasDefaultSourceForCategory(String category) {
        return defaultCategorySources.containsKey(category);
    }
    
    public static String getDefaultSourceForCategory(String category) {
        final String defCat = defaultForAllCategories;
        String catSpecificDef = defaultCategorySources.computeIfAbsent(category, (cat) -> { return SubstitutionTokenUtils.resolveSubstitutionTokens(defCat); });
        return catSpecificDef;
    }
    
}
