package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import org.lsst.ccs.bus.data.ConfigurationInfo;

/**
 * An object that describes an Agent's full configuration.
 * It contains:
 * - the Set of all the categories
 * - for each category it contains the corresponding CategoryTag object that
 *   describes all the tags applied for a given category.
 * - the GlobalName and its version
 * 
 * @author emarin
 */
public class ConfigurationDescription implements Serializable {
    
    private static final long serialVersionUID = 2262886830605244842L;
    
    @Deprecated
    public static final String DEFAULT_CONFIG_NAME = "";
    @Deprecated
    public static final String DEFAULT_INITIAL_CONFIG_NAME = "defaultInitial";
    @Deprecated
    public static final String DEFAULT_CAT = "";
    @Deprecated
    public static final String SAFE_CONFIG_NAME = "safe";
    
    /** A global name. */
    //TO-DO: decide how global name and its version are used.
    @Deprecated
    private String name = null;
    // The global version
    private int version = CategoryDescription.UNDEF_VERSION;
    
    private final Set<String> categories = new HashSet<>();
    
    private final Map<String,CategoryDescription> categoryTagsMap = new TreeMap<>();
    
    /**
     * Default empty constructor.
     * 
     */
    public ConfigurationDescription() {
    }


    /**
     * Build a ConfigurationDescription from the Set of categories.
     * 
     * @param categories The Set of the configuration categories.
     */
    public ConfigurationDescription(Set<String> categories) {
        this.categories.addAll(categories);
    }
    
    /**
     * Create a ConfigurationDescription object with all the provided categories
     * in the "safe" tag.
     * 
     * @param categories The categories that are to be in the defaultInitial tag.
     * 
     * @return The newly created ConfigurationDescription.
     */
    public static ConfigurationDescription initialDefaultConfiguration(Set<String> categories) {
        ConfigurationDescription res = new ConfigurationDescription(categories);
        return res.withDefaultInitial(true);
    }
    
    /**
     * Apply the provided category tags to the current ConfigurationDescription object.
     * 
     * @param taggedCategories The category tags to be applied.
     * 
     * @return The modified ConfigurationDescription.
     */
    public final ConfigurationDescription parseConfigurationString(String ... taggedCategories) {
        if (taggedCategories == null || taggedCategories.length == 0) {
            return this;
        }
        for (String s : taggedCategories){
            CategoryDescription categoryTag = CategoryDescription.parseCategoryTagInput(s);
            addCategoryTag(categoryTag);
        }
        return this;
    }

    public final ConfigurationDescription parseConfigurationDescription(String description) {
        String[] taggedCategories = description.split(",");
        if (taggedCategories == null || taggedCategories.length == 0) {
            return this;
        }
        for (String s : taggedCategories){
            CategoryDescription categoryTag = CategoryDescription.parseCategoryTagInput(s);
            addCategoryTag(categoryTag);
        }
        return this;
    }

    public void removeCategory(String cat) {
        categories.remove(cat);
        categoryTagsMap.remove(cat);
    }
    
    public ConfigurationDescription withDefaultInitial(boolean withDefaults) {
        if (withDefaults) {
            for(String cat : categories) {
                if(!categoryTagsMap.containsKey(cat)) {
                    putSourcedTagForCategory(cat, CategoryDescription.DEFAULT_INITIAL_CONFIG_NAME, SingleCategoryTag.getDefaultSourceForCategory(cat), CategoryDescription.DEFAULT_V_STRING);
                }
            }
        }
        return this;
    }

    public void addCategoryTag(CategoryDescription categoryTag) {
        String cat = categoryTag.getCategoryName();
        categories.add(cat);
        CategoryDescription newCat = new CategoryDescription(categoryTag.getCategoryName()).merge(categoryTag);
        newCat.setHasChanges(categoryTag.hasChanges);
        categoryTagsMap.put(cat, newCat);
    }
    
    
    @Deprecated
    public void putTagForCategory(String cat, String tag, String version) {
        putSourcedTagForCategory(cat, tag, null, version);
    }

    public void putSourcedTagForCategory(String cat, String tag, String source, String version) {
        CategoryDescription catTag = categoryTagsMap.get(cat);
        if ( catTag == null ) {
            catTag = new CategoryDescription(cat);
            categoryTagsMap.put(cat, catTag);
        }
        catTag.addOrUpdadateSingleTagVersion(source, tag, version);
    }
    
    public void putSingleTagForCategory(String cat, SingleCategoryTag sourcedTag, String version) {
        CategoryDescription catTag = categoryTagsMap.get(cat);
        if ( catTag == null ) {
            catTag = new CategoryDescription(cat);
            categoryTagsMap.put(cat, catTag);
        }
        catTag.addSingleTagVersion(sourcedTag, version, true);
    }

    /**
     * Get the Set of categories in this description.
     * @return The Set with the categories.
     */
    public Set<String> getCategoriesSet() {
        return Collections.unmodifiableSet(categoryTagsMap.keySet());
    }

    public CategoryDescription getCategoryTag(String category) {
        return categoryTagsMap.get(category);
    }
    
    public boolean isEmpty() {
        return categoryTagsMap.isEmpty();
    }
    
    public boolean containsCategory(String category) {
        return categoryTagsMap.containsKey(category);
    }
    
    public String getDescriptionName() {
        return "["+toString().replace("\n",",")+"]";
    }
    
    public String getFullDescriptionName() {
        return "["+convertToString(true).replace("\n",",")+"]";
    }

    public int getVersion() {
        return version;
    }

    public ConfigurationDescription cloneForCategories(Set<String> categories) {
        ConfigurationDescription result = new ConfigurationDescription(categories);
        for ( String category : categories ) {
            if ( containsCategory(category) ) {
                result.addCategoryTag(getCategoryTag(category));
            }
        }
        return result;
    }
    
    public ConfigurationDescription merge(ConfigurationDescription input) {
        for (String category : input.getCategoriesSet() ) {
            CategoryDescription inputCategoryTag = input.getCategoryTag(category);
            CategoryDescription thisCategoryTag = getCategoryTag(category);
            if ( thisCategoryTag == null ) {
                categories.add(category);
                addCategoryTag(inputCategoryTag);
            } else {
                for (SingleCategoryTag singleTag : inputCategoryTag.getSingleCategoryTags() ) {
                    thisCategoryTag.addOrUpdadateSingleTag(singleTag);
                }
            }
        }
        return this;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + Objects.hashCode(this.categoryTagsMap);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ConfigurationDescription other = (ConfigurationDescription) obj;
        return categoryTagsMap.equals(other.categoryTagsMap);
    }
    
    @Override
    public String toString() {
        return convertToString(false);
    }
    
    public String convertToString(boolean showDefaultSources) {
        if ( isEmpty() ) {
            return "";
        }
        StringBuilder sb = new StringBuilder(0); 
        boolean isFirst = true;
        for (Map.Entry<String, CategoryDescription> e : categoryTagsMap.entrySet()) {
            if ( !isFirst ) {
                sb.append("\n");                
            } else {
                isFirst = false;
            }
            CategoryDescription catTag = e.getValue();
            sb.append(catTag.convertToString(showDefaultSources));
        }
        return sb.toString();        
    }
    
    public static ConfigurationDescription fromConfigurationInfo(ConfigurationInfo ci, Set<String> categories) {        
        ConfigurationDescription res = new ConfigurationDescription();
        for ( String cat : ci.getConfigurationDescriptionObject().getCategoriesSet() ) {
            if ( categories.contains(cat) ) {
                res.addCategoryTag(ci.getConfigurationDescriptionObject().getCategoryTag(cat));
            }
        }
        return res;
    }

    public static ConfigurationDescription fromConfigurationInfo(ConfigurationInfo ci) {
        return fromConfigurationInfo(ci, ci.getCategorySet());
    }
    
    
    
}
