package org.lsst.ccs.bus.messages;

import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;


/**
 * Status message sent when Alerts are cleared.
 * It contains a String array of the alertIds that have been cleared
 * 
 * @author LSST CCS Team
 * 
 */
public final class StatusClearedAlert extends StatusAlert<String[], Object> {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 382404850729235L;
    
    /**
     * Build a StatusClearAlert from the provided String array and the
     * {@code RaisedAlertSummary} of the subsystem.
     * @param clearAlertIds Array of AlertIds that have been cleared.
     * @param raisedAlertSummary The summary of alerts at the time the object is
     * created.
     */
    public StatusClearedAlert(String[] clearAlertIds, RaisedAlertSummary raisedAlertSummary) {
        super(clearAlertIds, raisedAlertSummary);
    }
    
    /**
     * Get the String array of the alertIds that have been cleared.
     * 
     * @return The String array of with the ids of the alerts that have been cleared.
     */
    public String[] getClearAlertIds() {
        return getObject();
    }
    
    @Override
    protected Object encodeObject(String[] obj) {
        return null;
    }
    
    /**
     * Check if the Alert for the provided id was partially cleared.
     * 
     * @param id The Alert id we are checking
     * @return true if the provided id was partially cleared.
     */
    public boolean isPartialClear(String id) {
        for ( String clearedId : getClearAlertIds() ) {
            if ( clearedId.equals(id) ) {
                return getRaisedAlertSummary().getRaisedAlert(clearedId) != null;
            }
        }
        throw new IllegalArgumentException("Alert id = "+id+" was not cleared.");
    }
    
    /**
     * Get the AlertState for the provided alert id.
     * @param id The Alert id.
     * @return AlertState.NOMINAL if the alert was fully cleared or the Alert's highest severity (should be WARNING).
     */
    public AlertState getStateForClearedAlert(String id) {
        for ( String clearedId : getClearAlertIds() ) {
            if ( clearedId.equals(id) ) {
                RaisedAlertHistory history = getRaisedAlertSummary().getRaisedAlert(clearedId);
                return history == null ? AlertState.NOMINAL : history.getHighestAlertState();
            }
        }
        throw new IllegalArgumentException("Alert id = "+id+" was not cleared.");        
    } 
}
