package org.lsst.ccs.bus.data;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.config.CategoryDescription;
import org.lsst.ccs.config.ConfigurationDescription;
import org.lsst.ccs.config.SingleCategoryTag;
import org.lsst.ccs.utilities.structs.ParameterPath;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A Class containing the Agent's configuration information.
 *
 * This Class is published on the buses every time there is a configuration 
 * operation (save, load, drop or change) and it contains:
 * - the ConfigurationState
 * - a List with the paths of the parameters that have changed
 * - the timestamp of the configuration operation
 * - a List of all the ConfigurationParameterInfo (to be removed since it
 * duplicates the information in the data dictionary)
 * 
 * TO-DO: should we add the type of configuration operation? enum ConfigurationListener.ConfigurationOperation
 *
 * @author The LSST CCS Team
 */
public final class ConfigurationInfo implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 45966430232923784L;

    //Use corresponding fields in CategoryTag class.
    @Deprecated
    public static final int DEFAULT_VERSION = -1;
    @Deprecated
    public static final int LATEST_VERSION = -2;
    @Deprecated
    public static final int UNDEF_VERSION = -3;

    private ConfigurationState configState;
    private final List<String> recentChanges = new ArrayList<>();
    private final List<ConfigurationParameterInfo> parametersView = new ArrayList<>();
    private CCSTimeStamp ccsTimeStamp;
    private String configurationDescription;
    private String fullConfigurationDescription;
    private transient ConfigurationDescription cd;

    //TO-DO: should this be deprecated?
    @Deprecated
    private String descriptionName;
    
    
    //DEPRECATED: REMOVE NEXT MAJOR RELEASE WHEN BACKWARD INCOMPATIBLE CHANGES ARE INTRODUCED
    @Deprecated
    private final Map<String, Boolean> hasCategoryChanged = new HashMap<>();
    @Deprecated
    private final Map<String, String> tags = new HashMap<>();
    @Deprecated
    private final Map<String, Integer> versions = new HashMap<>();
    @Deprecated
    private String globalName;
    @Deprecated
    private Integer version = CategoryDescription.UNDEF_VERSION;
    
    
    /**
     * Returns the subsystem description name, ie the name of the description
     * file (groovy) the subsystem is started with.
     *
     * @return the subsystem description name
     */
    //TO-DO: should this be deprecated?
    @Deprecated
    public String getDescriptionName() {
        return descriptionName;
    }

    /**
     * Get the CCSTimeStamp of when this object was created.
     *
     * @return The CCSTimeStamp when this object was created.
     */
    public CCSTimeStamp getCCSTimeStamp() {
        return ccsTimeStamp;
    }

    /**
     * Returns the String representation of the ConfigurationDescription without the default sources.
     * This contains all the tags loaded for all the categories, their order, their
     * versions, if they have changes etc.
     *
     * This string is converted to a ConfigurationDescription object when using method
     * {@code #
     * 
     * @return The full ConfigurationDescription for this Agent
     */
    public String getConfigurationDescription() {
        return configurationDescription;
    }

    /**
     * Returns the String representation of the full ConfigurationDescription.
     * This contains all the tags loaded for all the categories, their order, their
     * versions, if they have changes etc.
     *
     * This string is converted to a ConfigurationDescription object when using method
     * {@code #
     * 
     * @return The full ConfigurationDescription for this Agent
     */
    public String getFullConfigurationDescription() {
        return fullConfigurationDescription != null ? fullConfigurationDescription : getConfigurationDescription();
    }

    /**
     * Returns true if there are unsaved changes for the specified category.
     *
     * @param category
     * @return true if there are unsaved changes for the specified category,
     * false otherwise.
     */
    public boolean hasChangesForCategory(String category) {
        return getConfigurationDescriptionObject().getCategoryTag(category).hasChanges();
    }

    /**
     * Returns true if there are changes in any of the categories.
     * 
     * @return false if there are no unsaved changes for all categories.
     */
    public boolean hasChanges() {        
        for (String cat : getCategorySet()) {
            if ( getConfigurationDescriptionObject().getCategoryTag(cat).hasChanges() ) {
                return true;
            }
        }
        return false;
    }

    /**
     * @param category
     * @return the base configuration name for the specified category.
     * @deprecated this method should be replaced by fetching the corresponding
     *             ConfigurationDescription with getConfigurationDescriptionObject()
     *             and then fetching the desired CategoryTag.
     */
    @Deprecated
    public String getConfigNameForCategory(String category) {
        return getConfigurationDescriptionObject().getCategoryTag(category).toString().replace(category+":", "");
    }

    /**
     * @param cat
     * @return the version of the configuration applied to the given category.
     * @deprecated this method should be replaced by fetching the corresponding
     *             ConfigurationDescription with getConfigurationDescriptionObject()
     *             and then fetching the desired CategoryTag.
     */
    @Deprecated
    public Integer getConfigVersion(String cat) {
        CategoryDescription catTag = getConfigurationDescriptionObject().getCategoryTag(cat);
        if ( catTag == null ) {
            return CategoryDescription.UNDEF_VERSION;
        }
        SingleCategoryTag tag0 = catTag.getSingleCategoryTags().get(0);
        Integer ver = -1;
        try {
            ver = Integer.parseInt(tag0.getVersion());
        } catch (Exception e) {            
        }
        return ver;
    }

    /**	
     * @return the String representation of the configuration state	
     * @deprecated use getConfigurationDescription instead	
     */	
    @Deprecated	
    public String getConfigurationName() {	
        return configurationDescription;
    }    
    
    /**
     * Tests if {@code category} is one of the subsystem parameter categories.
     *
     * @param category
     * @return true if {@code category} is one of the parameter categories of
     * the subsystem.
     */
    public boolean hasCategory(String category) {
        return getConfigurationDescriptionObject().containsCategory(category);
    }

    /**
     * @return the set of categories the configurable parameters of this {
     * @configurationInfo} are split into.
     */
    public Set<String> getCategorySet() {
        return getConfigurationDescriptionObject().getCategoriesSet();
    }

    /**
     * @return the configuration state
     */
    public ConfigurationState getConfigurationState() {
        return configState;
    }

    /**
     * Set the configuration state
     * @param state The ConfigurationState
     */
    public void setConfigurationState(ConfigurationState state) {
        this.configState = state;
    }

    /**
     * @return a String representation of the {
     * @ConfigurationInfo} object
     *
     * DO NOT MODIFY. IT IS USED BY COMMAND getConfigurationState in
     * ConfigurationService.java
     */
    @Override
    public String toString() {
        return "Configuration state : " + configState + ":" + getConfigurationDescription();
    }

    /**
     * Returns a list of {@code ConfigurationParameterInfo}. Each element of
     * this list represents the current view of a configurable parameter.
     *
     * @return a List of {@code ConfigurationParameterInfo}
     */
    public List<ConfigurationParameterInfo> getAllParameterInfo() {
        return Collections.unmodifiableList(parametersView);
    }

    /**
     * Return the ConfigurationParameterInfo of a given configurable parameter
     *
     * @param pathName the path name of a parameter, ie
     * {@code moduleName/parameterName}, old format
     * {@code moduleName//parameterName} is still supported but should be
     * replaced)
     * @return the ConfigurationParameterInfo for the provided path.
     */
    public ConfigurationParameterInfo getCurrentParameterInfo(String pathName) {
        return configByPath.get(convertToParameterPath(pathName).toString());
    }

    /**
     * Returns the current ConfigurationParameterInfo of each parameter that
     * belong to the specified category.
     *
     * @param category
     * @return a Map of pathName and the corresponding
     * ConfigurationParameterInfo.
     */
    public Map<String, ConfigurationParameterInfo> getCurrentParameterInfoForCategory(String category) {
        Set<String> pathsForCategory = getCurrentValuesForCategory(category).keySet();
        Map<String, ConfigurationParameterInfo> result = new HashMap<>();
        for (String path : pathsForCategory) {
            ConfigurationParameterInfo cpi = configByPath.get(path);
            result.put(path, cpi);
        }
        return result;
    }

    /**
     * Returns the ConfigurationParameterInfo of each parameter that belongs to
     * the specified configurable component.
     *
     * @param componentName the name of the component
     * @return a Map of parameter names and corresponding
     * ConfigurationParameterInfo.
     */
    public Map<String, ConfigurationParameterInfo> getCurrentParameterInfoFor(String componentName) {
        Set<String> pathsForComponent = getCurrentValuesFor(componentName).keySet();

        Map<String, ConfigurationParameterInfo> result = new HashMap<>();
        for (String path : pathsForComponent) {
            ConfigurationParameterInfo cpi = configByPath.get(path);
            result.put(path, cpi);
        }
        return result;
    }

    /**
     * Get the list of ConfigurationParameterInfo object whose value has changed
     * between the old and the new ConfigurationInfo. The returned
     * ConfigurationParameterInfo objects belong to the new
     * ConfigurationParameterInfo.
     *
     * @param ci The ConfigurationInfo object from which we want to calculate
     * the difference
     * @return The List of ConfigurationParameterInfo objects that have changed.
     */
    public List<ConfigurationParameterInfo> diff(ConfigurationInfo ci) {
        List<ConfigurationParameterInfo> result = new ArrayList();
        for (ConfigurationParameterInfo newParInfo : getAllParameterInfo()) {
            String newParValue = newParInfo.getCurrentValue();
            String oldParValue = ci.getCurrentValueForParameter(newParInfo.getPathName());
            if (newParValue == null) {
                if (oldParValue != null) {
                    result.add(newParInfo);
                }
            } else if (!newParValue.equals(oldParValue)) {
                result.add(newParInfo);
            }
        }
        return result;
    }

    /**
     * Returns a list of the last changes that occurred since the previous
     * {@code ConfigurationInfo} publication.
     * <ul>
     * <li>If a single change is processed, the returned list contains this
     * change only.
     * <li>If a bulk change is processed, the returned list contains all the
     * changes that have been submitted, validated and successfully set during
     * this bulk change session.
     * </ul>
     *
     * @return the list of recent changes.
     */
    public List<ConfigurationParameterInfo> getLatestChanges() {
        if (!complete) {
            throw new RuntimeException("Cannot use this ConfigurationInfo object; it's not complete yet.");
        }
        List<ConfigurationParameterInfo> latestChanges = new ArrayList<>();
        for (String path : recentChanges) {
            ConfigurationParameterInfo p = configByPath.get(path);
            if (p != null) {
                latestChanges.add(p);
            }
        }
        return latestChanges;
    }

    // Common actions on the list of parameter info
    /**
     * Returns the current value of each parameter that belong to the specified
     * category.
     *
     * @param category
     * @return a Map of parameter pathName and their current value.
     */
    public Map<String, String> getCurrentValuesForCategory(String category) {
        if (!complete) {
            throw new RuntimeException("Cannot use this ConfigurationInfo object; it's not complete yet.");
        }
        return Collections.unmodifiableMap(valuesMapByCategory.get(category));
    }

    /**
     * Returns the current value of each parameter that belong to the specified
     * configurable component.
     *
     * @param componentName the name of the component
     * @return a Map of parameter names to their current value.
     */
    public Map<String, String> getCurrentValuesFor(String componentName) {
        if (!complete) {
            throw new RuntimeException("Cannot use this ConfigurationInfo object; it's not complete yet.");
        }
        return Collections.unmodifiableMap(valuesMapByComponent.get(componentName));
    }

    /**
     * Return the current value of a given configurable parameter
     *
     * @param pathName the path name of a parameter, ie
     * {@code moduleName/parameterName}, old format
     * {@code moduleName//parameterName} is still supported but should be
     * replaced)
     * @return a String representation of the parameter's value
     */
    public String getCurrentValueForParameter(String pathName) {
        return configByPath.get(convertToParameterPath(pathName).toString()).getCurrentValue();
    }

    /**
     * TO-DO: Why are we going from path to ParameterPath and then back to path?
     * Is it just to make sure the format works?
     */
    private ParameterPath convertToParameterPath(String pathName) {
        if (!complete) {
            throw new RuntimeException("Cannot use this ConfigurationInfo object; it's not complete yet.");
        }
        if (pathName.contains("//")) {
            System.out.println("double slash should be removed when accessing the value of a parameter by its path");
        }
        ParameterPath pp = ParameterPath.valueOf(pathName);
        return pp;
    }
    
    /**
     * Get the ConfigurationDescription object describing the full configuration
     * for this Agent. This object is built on the fly using {@code getConfigurationDescription()}
     * 
     * @return The ConfigurationDescription object for this Agent.
     */
    public ConfigurationDescription getConfigurationDescriptionObject() {
        if ( cd == null ) {
            String desc = getFullConfigurationDescription().replace("[", "").replace("]", "");
            cd = new ConfigurationDescription().parseConfigurationDescription(desc);
        }
        return cd;
    }
    
    private Map<String, ConfigurationParameterInfo> configByPath;
    private Map<String, Map<String, String>> valuesMapByCategory;
    private Map<String, Map<String, String>> valuesMapByComponent;
    private boolean complete = false;

    private void complete() {
        complete = true;

        configByPath = new HashMap();
        valuesMapByCategory = new HashMap();
        valuesMapByComponent = new HashMap();

        for (ConfigurationParameterInfo par : parametersView) {
            String pathName = par.getPathName();
            String currentValue = par.getCurrentValue();
            String category = par.getCategoryName();
            String component = par.getComponentName();

            configByPath.put(pathName, par);

            Map<String, String> valuesMapForCategory = valuesMapByCategory.getOrDefault(category, new HashMap<>());
            valuesMapForCategory.put(pathName, currentValue);
            valuesMapByCategory.put(category, valuesMapForCategory);

            Map<String, String> valuesMapForComponent = valuesMapByComponent.getOrDefault(component, new HashMap<>());
            valuesMapForComponent.put(par.getParameterName(), currentValue);
            valuesMapByComponent.put(component, valuesMapForComponent);

        }
    }

    /**
     * Groups the ConfigurationParameterInfo objects of the input list by the
     * component they belong to.
     *
     * @param parmList
     * @return a Map of component name to the list of their parameters.
     */
    public static Map<String, List<ConfigurationParameterInfo>> getParameterInfoGroupByComponent(List<ConfigurationParameterInfo> parmList) {
        return parmList.stream()
                .collect(Collectors.groupingBy(ConfigurationParameterInfo::getComponentName, Collectors.toList()));
    }

    /**
     * Groups the ConfigurationParameterInfo objects of the input list by the
     * category they belong to.
     *
     * @param parmList
     * @return a Map of category names to the list of parameters that belong to
     * this category.
     */
    public static Map<String, List<ConfigurationParameterInfo>> getParameterInfoGroupByCategory(List<ConfigurationParameterInfo> parmList) {
        return parmList.stream()
                .collect(Collectors.groupingBy(ConfigurationParameterInfo::getCategoryName, Collectors.toList()));
    }

    private void readObject(java.io.ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        if (!complete) {
            complete();
        }
    }

    /**
     * A builder for a {
     *
     * @ConfigurationInfo} object.
     */
    public static class Builder {

        private final ConfigurationInfo ci;

        private final Map<ParameterPath, ConfigurationParameterInfo.Builder> parameterBuilders = new HashMap<>();

        public Builder(ConfigurationInfo ci) {
            this.ci = ci;
        }

        public Builder() {
            ci = new ConfigurationInfo();
        }

        public Builder setDescription(String description) {
            ci.descriptionName = description;
            return this;
        }

        public Builder setConfigurationState(ConfigurationState configState) {
            ci.configState = configState;
            return this;
        }

        public Builder addParameter(ParameterPath path, Type type, String category, String description, boolean isFinal, boolean isReadOnly, boolean isBuild) {
            ConfigurationParameterInfo.Builder builder = new ConfigurationParameterInfo.Builder();
            builder.addParameter(path, type, category, description, isFinal, isReadOnly, isBuild);
            parameterBuilders.put(path, builder);
            ci.parametersView.add(builder.build());
            return this;
        }

        public Builder updateParameter(ParameterPath path, String configuredValue, String currentValue, boolean dirty) {
            parameterBuilders.get(path).updateParameter(configuredValue, currentValue, dirty);
            return this;
        }

        public Builder addRecentChange(String path) {
            ci.recentChanges.add(path);
            return this;
        }

        public Builder updateConfigurationDescription(String configurationDescription) {
            return updateConfigurationDescription(configurationDescription,configurationDescription);
            
        }
        public Builder updateConfigurationDescription(String fullConfigurationDescription, String configurationDescription) {
            this.ci.fullConfigurationDescription = fullConfigurationDescription;
            this.ci.configurationDescription = configurationDescription;
            this.ci.cd = null;
            return this;
        }

        public Builder setCCSTimeStamp(CCSTimeStamp ccsTimeStamp) {
            ci.ccsTimeStamp = ccsTimeStamp;
            return this;
        }

        public ConfigurationInfo build() {
            ci.complete();
            return ci;
        }

    }

}
