package org.lsst.ccs.bus.data;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.BitSet;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Generic Data class for Key-Value pairs.
 * This class is meant to be used by CCS Agents to publish key-value pairs type
 * information. Subclasses of this class will provide additional information when
 * needed. For example KeyValueTimeData provides a timestamp for the key-value pair.
 * 
 * @author The LSST CCS Team
 *
 */
public class KeyValueData implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -45918237359438L;
    
    private final String key;
    private final Serializable value;
    private final CCSTimeStamp ccsTimeStamp;
    
    protected transient boolean needsEncoding = false;

    public static enum KeyValueDataType { KeyValueTrendingData, 
    @Deprecated
    KeyValuePlotData, KeyValueMetaData };
    private final KeyValueData.KeyValueDataType type;

    
    /**
     * Build a KeyValueData object providing the name and the value for the key-value pair
     * and the CCSTimeStamp in which this data was accumulated and their type.
     * The type defines what client codes will do with the KeyValueData.
     * 
     * @param key The String representing the key
     * @param value The Object representing the value of the pair.
     * @param ccsTimeStamp The CCSTimestamp for this KeyValueData.
     * @param type The type for this KeyValueData
     */
    public KeyValueData(String key, Serializable value, CCSTimeStamp ccsTimeStamp, KeyValueData.KeyValueDataType type) {
        this.key = key;
        this.value = value;
        this.type = type;
        this.ccsTimeStamp = ccsTimeStamp;        
        updateNeedsEncoding(value);        
    }
    
    /**
     * Build a KeyValueData object providing the name and the value for the key-value pair
     * and the CCSTimeStamp in which this data was accumulated.
     * 
     * @param key The String representing the key
     * @param value The Object representing the value of the pair.
     * @param ccsTimeStamp The CCSTimestamp for this KeyValueData.
     */
    public KeyValueData(String key, Serializable value, CCSTimeStamp ccsTimeStamp) {
        this(key,value,ccsTimeStamp,KeyValueDataType.KeyValueTrendingData);
    }

    /**
     * Build a KeyValueData object providing the name and the value for the key-value pair.
     * The timestamp is automatically assigned the time this object was created.
     * 
     * @param key The String representing the key
     * @param value The Object representing the value of the pair.
     */
    public KeyValueData(String key, Serializable value) {
        this(key, value, CCSTimeStamp.currentTime());
    }

    /**
     * Get the CCSTimeStamp associated with the key-value pair.
     * @return The CCSTimeStamp.
     */
    public CCSTimeStamp getCCSTimeStamp() {
        return ccsTimeStamp;
    }
    
    
    /**
     * Get the value embedding in this object.
     * @return The value.
     */
    public Serializable getValue() {
        return value;
    }

    /**
     * Get the key.
     * @return The name of the Key
     */
    public String getKey() {
        return key;
    }

    /**
     * Get the KeyValueData type.
     * @return The KeyValueData type.
     * 
     */
    public KeyValueDataType getType() {
        return type;
    }
    
    @Override
    public String toString() {
        return "KeyValueData{ key=" + key + ", value=" + value + ", timestamp=" + ccsTimeStamp+ ", type="+type+" }";
    }


    //Update the internal flag to decide if this KeyValueData needs to be
    //encoded before it is sent over the buses.
    //The flag is updated to only if the object does not currently need
    //encoding and if the contained object is not a basic type.    
    protected final void updateNeedsEncoding(Object obj) {
        if ( ! needsEncoding ) {
            if (obj != null) {
                Class clazz = obj.getClass();
                if (!clazz.isPrimitive()) {
                    if ( ! SKIP_ENCODING_TYPES.contains(clazz) ) {
                        needsEncoding = true;
                    }
                }
            }
        }
    }
    
    public boolean needsEncoding() {
        return needsEncoding;
    }
    
    private transient static final List<Class> SKIP_ENCODING_TYPES = Arrays.asList( new Class[] {
        Boolean.class,
        Short.class,
        Integer.class,
        Float.class,
        Double.class,
        Long.class,
        String.class,
        byte[].class,
        boolean[].class,
        short[].class,
        int[].class,
        float[].class,
        double[].class,
        long[].class,
        String[].class,
        BitSet.class,
        BigDecimal.class
    });
    
}
