package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An object designed to group together KeyValueData objects before publication
 * on the Status bus.
 * This is a KeyValueData itself, whose value is a List<KeyValueData>
 * At construction time this list is associated a time stamp, either as a user
 * input or set to the creation time by default.
 * When data is added to this list, it is stored internally as a KeyValueData.
 * If no time stamp is provided at the time the key-value pair is added, it is
 * by default assigned the time stamp of the list.
 * 
 * @author The LSST CCS Team
 */
public final class KeyValueDataList extends KeyValueData implements Serializable, Iterable<KeyValueData> {

    
    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 56498923019235263L;

    /**
     * A Map containing attributes for this data list.
     * This map was added to address https://jira.slac.stanford.edu/browse/LSSTCCS-2206
     */
    private final Map<String,Serializable> attributes = new HashMap<>();
    
    /**
     * Create a KeyValueDataList without a key or a timestamp.
     * By default the key is assigned the empty String, while the time stamp is
     * that of the object creation.
     * The value of this KeyValueData is a {@code List<KeyValueData>}
     * 
     */
    public KeyValueDataList() {
        this("");
    }
    
    /**
     * Create a KeyValueDataList with attributes.
     * By default the key is assigned the empty String, while the time stamp is
     * that of the object creation.
     * The value of this KeyValueData is a {@code List<KeyValueData>}
     * 
     * @param attributes The Map of attributes
     */
    public KeyValueDataList(Map<String,Serializable> attributes) {
        this("", attributes);
    }

    /**
     * Create a KeyValueDataList with a key.
     * The time stamp is by default the object creation time.
     * The value of this KeyValueData is a {@code List<KeyValueData>}
     * 
     * @param key The list's key
     */
    public KeyValueDataList(String key) {
        this(key,CCSTimeStamp.currentTime());
    }

    /**
     * Create a KeyValueDataList with a key.The time stamp is by default the object creation time.
     * The value of this KeyValueData is a {@code List<KeyValueData>}
     * 
     * @param key The list's key
     * @param attributes The Map of attributes
     */
    public KeyValueDataList(String key, Map<String,Serializable> attributes) {
        this(key,CCSTimeStamp.currentTime(), attributes);
    }

    /**
     * Create a KeyValueDataList with a key and a time stamp.
     * The value of this KeyValueData is a {@code List<KeyValueData>}
     * 
     * @param key The list's key
     * @param ccsTimeStamp The key-value pair ccsTimeStamp.
     */
    public KeyValueDataList(String key, CCSTimeStamp ccsTimeStamp) {
        super(key, new ArrayList<KeyValueData>(), ccsTimeStamp);
        this.needsEncoding = false;
    }

    /**
     * Create a KeyValueDataList with a key and a time stamp.
     * The value of this KeyValueData is a {@code List<KeyValueData>}
     * 
     * @param key The list's key
     * @param ccsTimeStamp The key-value pair ccsTimeStamp.
     * @param attributes The Map of attributes
     */
    public KeyValueDataList(String key, CCSTimeStamp ccsTimeStamp, Map<String,Serializable> attributes) {
        super(key, new ArrayList<KeyValueData>(), ccsTimeStamp);
        this.needsEncoding = false;
        this.attributes.putAll(attributes);
    }

    /**
     * Create a KeyValueDataList with a ccs time stamp.
     * The key is assigned by default the empty String. 
     * The value of this KeyValueData is a {@code List<KeyValueData>}
     * 
     * @param ccsTimeStamp The key-value pair ccs timestamp.
     * 
     */
    public KeyValueDataList(CCSTimeStamp ccsTimeStamp) {
        this("",ccsTimeStamp);
    }
    
    /**
     * Get the value of the provided attribute name.
     * 
     * @param name The name of the attribute
     * @return     The attribute's value. null if there is no such attribute.
     */
    public Serializable getAttribute(String name) {
        return attributes.get(name);
    }
    
    
    /**
     * Add a key-value pair to this KeyValueDataList by assigning it the
     * timestamp of the KeyDataList.
     * Internally the pair is stored in a KeyValueData object.
     * The default type is KeyValueData.KeyValueDataType.KeyValueTrendingData
     * 
     * @param key The key of the pair.
     * @param value The value of the pair.
     */
    public void addData(String key, Serializable value) {
        if ( value instanceof KeyValueDataList || value instanceof KeyValueData ) {
            throw new IllegalArgumentException("Data cannot be DataList or KeyValueData");
        } else {
            addData( new KeyValueData(key,value, getCCSTimeStamp()) );
        }
    }

    /**
     * Add a key-value pair to this KeyValueDataList by assigning it the
     * timestamp of the KeyDataList.
     * Internally the pair is stored in a KeyValueData object.
     * 
     * @param key The key of the pair.
     * @param value The value of the pair.
     * @param type The type of the KeyValueData to add.
     */
    public void addData(String key, Serializable value, KeyValueData.KeyValueDataType type) {
        addData(key,value,CCSTimeStamp.currentTime(),type);
    }

    /**
     * Add a key-value pair with a CCS timestamp to this KeyValueDataList.
     * Internally the pair is stored in a KeyValueData object.
     * The default type is KeyValueData.KeyValueDataType.KeyValueTrendingData
     * @param key The key of the pair.
     * @param value The value of the pair.
     * @param ccsTimeStamp The ccsTimeStamp associated to the pair.
     */
    public void addData(String key, Serializable value, CCSTimeStamp ccsTimeStamp) {
        addData(key,value,ccsTimeStamp,KeyValueData.KeyValueDataType.KeyValueTrendingData);
    }

    /**
     * Add a key-value pair with a CCS timestamp to this KeyValueDataList.
     * Internally the pair is stored in a KeyValueData object.
     * @param key The key of the pair.
     * @param value The value of the pair.
     * @param ccsTimeStamp The CCS timestamp associated to the pair.
     * @param type The type of the KeyValueData to add.
     */
    public void addData(String key, Serializable value, CCSTimeStamp ccsTimeStamp, KeyValueData.KeyValueDataType type) {
        if ( value instanceof KeyValueData ) {
            throw new IllegalArgumentException("Value cannot be KeyValueData, please use method addData(KeyDataValue).");
        } else {
            addData( new KeyValueData(key,value,ccsTimeStamp, type) );
        }
    }

    /**
     * Get the list of KeyValueData contained in this object.
     * @return The list of KeyValueData.
     */
    public List<KeyValueData> getListOfKeyValueData() {
        return getListOfKeyValueData(null);
    }
    
    /**
     * Get the list of KeyValueData contained in this object for a given type.
     * @param type The KeyValueDataType for which the data is required.
     * @return The list of KeyValueData that match the provided type.
     */
    public List<KeyValueData> getListOfKeyValueData(KeyValueData.KeyValueDataType type) {
        ArrayList<KeyValueData> list = new ArrayList<>();
        for ( KeyValueData d : (List<KeyValueData>)getValue() ) {
            if ( type == null || type == d.getType() ) {
                list.add(d);
            }            
        }            
        return list;
    }

    
    /**
     * Add a KeyValueData object to the KeyValueDataList.
     * @param data The KeyValueData object to add to the KeyValueDataList.
     */
    public void addData(KeyValueData data) {
        ((ArrayList<KeyValueData>)getValue()).add(data);
        if ( data != null && ! needsEncoding ) {
            needsEncoding = data.needsEncoding();
        }
    }                
    
    @Override
    public Iterator<KeyValueData> iterator() {
        return getListOfKeyValueData().iterator();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        boolean isFirst = true;
        for ( KeyValueData d : this ) {
            if (! isFirst ) {
                sb.append(", ");
            } else {
                isFirst = false;
            }
            if ( d == null ) {
                sb.append("null");
            } else {
                sb.append(d.getKey()).append("=").append(d.getValue());
            }
        }
        return sb.toString();
    }

}
