package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * Base class for all Alerts.
 * This class provides an alertId field that is used to uniquely identify similar
 * Alert instances: if two Alert instances have the same alertId, they are equal.
 * 
 * Alerts are raised when abnormal situations occur, and they can be raised at
 * a different AlertState level.
 * 
 * Additionally the Alert class has the following field:
 * <ul>
 *  <li>description: the description of the Alert</li>
 * </ul>
 * 
 * Since this class is final it must be used as is by subsystems. This class
 * provides a way to add key-value pairs so that subsystem developers can add
 * subsystem specific information to specific alerts. 
 * The values added to the Alert must be basic Serializable object or de-serialization
 * problems will arise in the receiving clients.
 * The Alert data can be provided only at construction time and cannot be modified.
 * 
 * When an Alert is raised a StatusAlert message will be published on the Status 
 * bus.
 * 
 * @author The LSST CCS Team
 */
public final class Alert implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -568951332963342L;

    private final String alertId, description; 
    
    private final Map<String,Object> innerData;
    
    /**
     * Base constructor for all Alerts.
     * 
     * @param alertId The alertId for this Alert instance.
     * @param description The Alert description
     */
    public Alert(String alertId, String description) {
        this(alertId, description, Collections.EMPTY_MAP);
    }

    /**
     * Base constructor for all Alerts.
     * 
     * @param alertId The alertId for this Alert instance.
     * @param description The Alert description
     * @param innerData The Alert data.
     */
    public Alert(String alertId, String description, Map<String,Object> innerData) {
        if ( innerData == null ) {
            innerData = Collections.EMPTY_MAP;
        }
        this.alertId = alertId;
        this.description = description;
        this.innerData = Collections.unmodifiableMap(innerData);
    }
    
    /**
     * Get the Alert id
     * @return The Alert id
     */
    public String getAlertId() {
        return alertId;
    }

    /**
     * Get the Alert description
     * @return The Alert description
     */
    public String getDescription() {
        return description;
    }
    
    /**
     * Get data contained in this Alert.
     * 
     * @param key The key with which this data was stored.
     * @return The data for the given key.
     */
    public Object getAlertData(String key) {
        return innerData.get(key);
    }
    
    /**
     * Get the Set of names for the Alert data.
     * @return The Set of Alert data names
     */
    public Set<String> getAlertDataNames() {
        return innerData.keySet();
    }

    @Override
    public String toString() {
        return "Alert "+getAlertId()+" ("+getDescription()+")";
    }


    @Override
    public int hashCode() {
        int hash = 7;
        hash = 13 * hash + Objects.hashCode(this.alertId);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (! (obj instanceof Alert) ) {
            return false;
        }
        final Alert other = (Alert) obj;
        return Objects.equals(this.alertId, other.alertId);
    }
    
}
