package org.lsst.ccs.bus.messages;

import java.io.Serializable;
import org.lsst.ccs.bus.states.StateBundle;

/**
 * Base class for all messages sent on the Status Bus.
 *
 * All StatusMessage objects contain the internal state of the Agent from which
 * they originated at the time of publication.
 *
 * The default encoding for StatusMessages breaks down the embedded object in
 * key-value pairs (and possibly timestamp) and returns the DataList object containing them.
 * Subclasses can change this default encoding behavior. Please refer to their
 * documentation for further details on their encoding strategy.
 *  
 * @author LSST CCS Team
 * @param <T> Any Serializable class embedded in the StatusMessage
 * @param <D> The class of the encoded object.
 * 
 */
public abstract class StatusMessage<T extends Serializable,D> extends BusMessage<T,D> {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -2133326596736909L;

    private StateBundle state;

    /**
     * Build a StatusMessage containing an already serialized object.
     *
     * @param clazz The Class of the serialized object.
     * @param ser The serialized version of the object to be sent over the buses.
     */
    public StatusMessage(Class clazz, byte[] ser) {
        super(clazz, ser);
    }


    /**
     * Build a StatusMessage from the provided Object.
     *
     * @param obj The Serializable object to be sent over the buses.
     */
    public StatusMessage(T obj) {
        super(obj);
    }
      
    /**
     * Set the StateBundle of this StatusMessage. The StateBundle represents the
     * state of the Agent at the time the message was sent on the buses.
     * 
     * Implementation Note. This method is meant to be used by the messaging
     * layer at the time the message is sent.
     * 
     * This method contains a protection against being invoked more than once.
     * 
     * @param state The StateBundle representing the state of the Agent at
     *              the time the message was sent.
     */
    public void setState(StateBundle state) {
        if ( this.state != null ) {
            throw new RuntimeException("SEVERE: the setState method must be invoked only once!");
        }
        this.state = state;        
    }    
    
    
    /**
     * Get the published State object that contains the Agent's internal state
     * at the moment the StatusMessage was published.
     *
     * @return The StateBundle object.
     */
    public StateBundle getState() {
        return state;
    }

    @Override
    public String toString() {
        String toString = super.toString();
        toString += " state=" + state + "\n";
        return toString;
    }

}
