package org.lsst.ccs.bus.states;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Contains a set of subsystem states, each represented by {@code Enum}. 
 * States are stored internally in a {@code ConcurrentHashMap}, mapped by their class names.
 * 
 * It also allows to define component level states in internal {@StateBundle} objects
 * stored in a {@code ConcurrentHashMap}, mapped by the component's name.
 *
 * This class is meant to be published on the buses with every status message
 * and when there is any state change.
 * 
 * This class is thread safe to the extent supported by the underlying {@code ConcurrentHashMap}:
 * no concurrent modifications are thrown, but atomicity of operations that involve
 * reading or modifying more than one elemental state is not guaranteed.
 *
 * @author The LSST CCS Team
 */
public class StateBundle implements Serializable, Cloneable {

    private static final long serialVersionUID = 19387419697469780L;
    
    private Map<String, Enum> allStates = new ConcurrentHashMap<>();

    private Map<String, StateBundle> componentStates = new ConcurrentHashMap<>();
    
    private transient static final Map<String,Class> classMap = new ConcurrentHashMap<>();
    private transient static final Map<String,Method> methodMap = new ConcurrentHashMap<>();
    private transient static final Map<String,Object> enumMap = new ConcurrentHashMap<>();

    /** 
     * Should remain empty on the sending side. It is filled if needed during
     * deserialization. It does not participate in any logic made on the 
     * StateBundle object (diff or merge).
     */
    private transient Map<String, String> subsystemSpecificStates = new HashMap<>();    
    
    //To be removed next backward incompatible release: LSSTCCS-1854
    @Deprecated
    private CCSTimeStamp lastModifiedCCSTimeStamp = CCSTimeStamp.currentTime();

    /** Constructs a StateBundle from a set of states.
     * @param states The set of States to construct the StateBundle with.
     */
    public StateBundle(Enum... states) {
        for (Enum state : states) {
            innerSetState(state);
        }
        updateLastModified();
    }

    /** Copy constructor. */
    private StateBundle(StateBundle b) {
        this.allStates.putAll(b.allStates);
        this.subsystemSpecificStates.putAll(b.subsystemSpecificStates);
        this.lastModifiedCCSTimeStamp = b.lastModifiedCCSTimeStamp;
        this.componentStates.putAll(b.componentStates);
    }

    /**
     * Merge the content of two StateBundle objects returning an updated cloned
     * version of the original StateBundle object.
     * The original StateBundle is left unchanged.
     * 
     * @param newState The StateBundle with the changes.
     * @return An update cloned version of the original StateBundle.
     *
     */
    public StateBundle mergeState(StateBundle newState) {
        StateBundle clone = this.clone();
        if ( newState != null ) {
            clone.allStates.putAll(newState.allStates);
            for (Entry<String, StateBundle> e : newState.componentStates.entrySet()) {
                StateBundle oldComponentState = clone.getComponentStateBundle(e.getKey());
                clone.componentStates.put(e.getKey(), oldComponentState.mergeState(e.getValue()));
            }
        }
        clone.updateLastModified();
        return clone;
    }

    /**
     * Finds what has changed.
     * 
     * @param oldState The StateBundle of the previous state .
     * @return a new StateBundle containing only the changed states.
     */
    public StateBundle diffState(StateBundle oldState) {
        if ( oldState == null ) {
            return new StateBundle(this);
        }
        
        StateBundle diff = new StateBundle();
        for (Enum e : allStates.values()) {
            if (!oldState.isInState(e)) {
                diff.innerSetState(e);
            }
        }
        
        for ( Entry<String, StateBundle> e : componentStates.entrySet() ) {
            StateBundle oldComponentState = oldState.getComponentStateBundle(e.getKey());
            StateBundle componentStateDiff = e.getValue().diffState(oldComponentState);
            diff.componentStates.put(e.getKey(), componentStateDiff);
        }
        diff.updateLastModified();
        return diff;
    }

    /**
     * Sets the specified state to the bundle of the given component, 
     * replacing any previous state of the same type.
     * @param component The component for which to set the state
     * @param states The states to set on the StateBundle.
     */
    public final void setComponentState(String component, Enum... states) {
        innerSetComponentState(component, states);
        updateLastModified();
    }
    
    /**
     * Sets specified states to the bundle, replacing any previous states of the same types. 
     * @param states The specified states to set on the StateBundle
     */
    public final void setState(Enum... states) {
        for (Enum state : states) innerSetState(state);
        updateLastModified();
    }
    
    //Sets the state without updating the timestamp of the last update.
    private void innerSetState(Enum state) {
        if ( state == null ) {
            throw new IllegalArgumentException("A state cannot be set to null");
        }
        allStates.put(state.getClass().getSimpleName(), state);
    }
    
    //Sets the component state without updating the timestamp of the last update.
    private void innerSetComponentState(String component, Enum... states) {        
        StateBundle componentStateBundle = getComponentStateBundle(component);
        componentStateBundle.setState(states);
    }

    /**
     * Returns the current value for the specified State.
     *
     * @param <T> An {@code Enum} that defines a set of states.
     * @param enumClass The class of the {@code Enum} representing the state.
     * @return The current value for the given state.
     */
    public final <T extends Enum<T>> T getState(Class<T> enumClass) {
        return (T) allStates.get(enumClass.getSimpleName());
    }
    
    /**
     * Returns the current value for the specified State of the given component.
     *
     * @param <T> An {@code Enum} that defines a set of states.
     * @param component The component for which the state is requested
     * @param enumClass The class of the {@code Enum} representing the state.
     * @return The current value for the given state for the provided component.
     */
    public final <T extends Enum<T>> Enum getComponentState(String component, Class<T> enumClass) {
        StateBundle sb = componentStates.get(component);
        return sb == null ? null : sb.getState(enumClass);
    }

    /**
     * Check if this {@code StateBundle} contains the specified state.
     *
     * @param <T> An {@code Enum} that defines a set of states.
     * @param state The value of the {@code Enum} to check.
     * @return true if this {@code StateBundle} contains the provided state
     */
    public <T extends Enum<T>> boolean isInState(T state) {
        Enum e = getState(state.getClass());
        return e == null ? false : e.equals(state);
    }

    /**
     * Check if the given component is in the specified state.
     *
     * @param <T> An {@code Enum} that defines a set of states.
     * @param component The component to be tested
     * @param state The value of the {@code Enum} to check.
     * @return true if the component is in the provided state
     */
    public <T extends Enum<T>> boolean isComponentInState(String component, T state) {
        Enum e = getComponentState(component, state.getClass());
        return e == null ? false : e.equals(state);
    }
    
    /**
     * Check if this component is in all the states of a given StateBundle.
     * 
     * @param component The component to be tested
     * @param stateBundle The StateBundle containing all the states to check.
     * @return true if this StateBundle is in all the states of the provided one.
     */
    public boolean isComponentInState(String component, StateBundle stateBundle) {
        StateBundle componentStateBundle = componentStates.get(component);
        if ( componentStateBundle == null ) {
            return false;
        }
        return componentStateBundle.isInState(stateBundle);
    }
    
    /**
     * Check if this StateBundle is in all the states of a given StateBundle.
     * 
     * @param stateBundle The StateBundle containing all the states to check.
     * @return true if this StateBundle is in all the states of the provided one.
     */
    public boolean isInState(StateBundle stateBundle) {
        for (Enum e : stateBundle.allStates.values()) {
            if (!isInState(e)) {
                return false;
            }
        }
        for (Entry<String,StateBundle> entry : stateBundle.componentStates.entrySet() ) {
            if (!isComponentInState(entry.getKey(), entry.getValue())) {
                return false;
            }
        }        
        return true;
    }

    /**
     * Get the Set of all internal components.
     * The set contains the name of the components.
     * 
     * @return The Set of components that have a state.
     */
    public Set<String> getComponentsWithStates() {
        return componentStates.keySet();
    }
    
    /**
     * For the given state, get the map of components/states.
     * The key in the map is the name of the component.
     *
     * @param <T> An {@code Enum} that defines a set of states.
     * @param enumClass The class of the {@code Enum} representing the state.
     * @return The map of components and their current state.
     */
    public final <T extends Enum<T>> Map<String,T> getComponentsWithState(Class<T> enumClass) {
        Map<String,T> result = new HashMap<>();
        for ( Entry<String,StateBundle> entry : componentStates.entrySet() ) {
            T e = entry.getValue().getState(enumClass);
            if ( e != null ) {
                result.put(entry.getKey(), e);
            }
        }
        return result;
    }

    /**
     * Get the StateBundle for the given component.
     * @param component The name of the component.
     * @return the {@code StateBundle} for the given component
     */
    public final synchronized StateBundle getComponentStateBundle(String component) {
        if ( component == null || component.isEmpty() ) {
            return this;
        }
        StateBundle componentStateBundle = componentStates.get(component);
        if ( componentStateBundle == null ) {
            componentStateBundle = new StateBundle();
            componentStates.put(component, componentStateBundle);
        }
        return componentStateBundle;
    }
    
    //To be removed next backward incompatible release: LSSTCCS-1854
    @Deprecated
    public CCSTimeStamp getLastModifiedCCSTimeStamp() {
        return lastModifiedCCSTimeStamp;
    }
    
    /**
     * Get the Map of all the states that were decoded when this message
     * was received, i.e. all the states for which the client had the
     * class definition. All the other states can be fetched via the getInternalStates() method.
     * 
     * @return The map of all the decoded states.
     */
    public Map<String,Enum> getDecodedStates() {
        return (Map<String,Enum>)Collections.unmodifiableMap(allStates);
    }
    
    /**
     * Get the Map of all the states in this StateBundle.
     * Since it is not guaranteed that all states were decoded upon receipt 
     * we convert all the state Enums to Strings.
     * 
     * @return The map of all the states as strings.
     */
    public Map<String,String> getAllStatesAsStrings() {
        Map<String,String> allStatesAsStrings = new TreeMap<>();
        for ( Entry<String,Enum> entry : allStates.entrySet() ) {
            allStatesAsStrings.put(entry.getKey(), entry.getValue().toString());
        }
        allStatesAsStrings.putAll(subsystemSpecificStates);
        return allStatesAsStrings;
    }

    /**
     * Gets string representations of the internal states.
     * @return a map of enum class name to the string value of this enum.
     */
    public Map<String, String> getInternalStates() {
        return Collections.unmodifiableMap(subsystemSpecificStates);
    }
    
    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof StateBundle)) {
            return false;
        }
        StateBundle state = (StateBundle) o;
        if ( allStates.size() != state.allStates.size() ) {
            return false;
        }
        for (String c : allStates.keySet()) {
            if (!state.isInState(allStates.get(c))) {
                return false;
            }
        }
        if ( componentStates.size() != state.componentStates.size() ) {
            return false;
        }
        for (String c : state.componentStates.keySet()) {
            if ( !state.componentStates.get(c).equals(componentStates.get(c)) ) {
                return false;
            }
        }
        
        return true;
    }

    @Override
    public int hashCode() {
        int result = 0;
        for (Enum e : allStates.values()) {
            result = 31 * result + e.hashCode();
        }
        for (StateBundle sb : componentStates.values()) {
            result = result + sb.hashCode();
        }
        return result;
    }

    @Override
    public StateBundle clone() {
        StateBundle b = new StateBundle(this);
        return b;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        for ( Entry<String,String> e : getAllStatesAsStrings().entrySet() ) {
            if ( ! e.getValue().isEmpty() ) {
                sb.append(e.getKey()).append(":").append(e.getValue()).append(" ");                
            }
        }
        sb.append("\n");
        
        Map<String,StateBundle> compStates = new TreeMap<>(componentStates);
        for ( Entry<String,StateBundle> e : compStates.entrySet() ) {
            String bundleStr = e.getValue().toString();
            if ( ! bundleStr.isEmpty() ) {
                sb.append(e.getKey()).append(": ").append(bundleStr);
            }
        }
        sb.append("\n");
        return sb.length() > 0 ? sb.substring(0, sb.length() - 1) : "";
    }

    private void updateLastModified() {        
        lastModifiedCCSTimeStamp = CCSTimeStamp.currentTime();
    }
    
    /**
     * all the states (framework and subsystem specific) are written in the same
     * map.
     */
    private void writeObject(ObjectOutputStream out) throws IOException {
        out.writeInt(allStates.size()+subsystemSpecificStates.size());
        for (Enum e : allStates.values()) {
            out.writeObject(e.getClass().getName());
            out.writeObject(e.toString());
        }
        for (Map.Entry<String, String> entry : subsystemSpecificStates.entrySet()) {
            out.writeObject(entry.getKey());
            out.writeObject(entry.getValue());
        }
        out.writeInt(componentStates.size());
        for ( Entry<String, StateBundle> entry : componentStates.entrySet() ) {
            out.writeObject(entry.getKey());
            entry.getValue().writeObject(out);
        }
        out.writeObject(getLastModifiedCCSTimeStamp());
    }

    /** For enums the class of which is locally unknown, their class name and value
     are stored in the subsystemSpecificStates map. */
    private void readObject(ObjectInputStream in) throws IOException,
            ClassNotFoundException {
        int n = in.readInt();
        allStates = new ConcurrentHashMap<>();
        subsystemSpecificStates = new ConcurrentHashMap<>();
        for (int i = 0; i < n; i++) {
            String className = (String) in.readObject();
            String value = (String) in.readObject();

            String enumKey = className+":"+value;
            try {
                Object e = enumMap.get(enumKey);
                if ( e == null ) {
                    Method m = methodMap.get(className);
                    if ( m == null ) {
                        Class<Enum> c = classMap.get(className);
                        if ( c == null ) {
                            c = (Class<Enum>) Class.forName(className);
                            classMap.put(className, c);
                        }
                        m = c.getDeclaredMethod("valueOf", String.class);
                        methodMap.put(className, m);
                    }
                    e = (Enum) m.invoke(null, value);
                    enumMap.put(enumKey, e);
                } 
                
                Class<Enum> c = classMap.get(className);
                if ( e instanceof String ) {
                    //In this case the Class was not found, and we should have cached
                    //the String representation of the enum
                    subsystemSpecificStates.put(className, value);
                } else {                
                    allStates.put(c.getSimpleName(), (Enum)e);
                }
            } catch (ClassNotFoundException | IllegalAccessException | IllegalArgumentException | NoSuchMethodException | SecurityException | InvocationTargetException e) {
                // Store as a pair of String in subsystemSpecificStates.
                subsystemSpecificStates.put(className, value);
                //Cache the value of the enum to avoid invoking Class.forName for
                //Enum classes that are not on the classpath
                enumMap.putIfAbsent(enumKey, value);
            }

        }

        int nComponents = in.readInt();        
        //The following try-catch block is to keep backward compatibility with
        //previous versions of the toolkit that don't have internal component
        //states. If an exception is thrown we quit.
        componentStates = new ConcurrentHashMap<>();

        for ( int i = 0; i< nComponents; i++ ) {
                String componentName = (String) in.readObject();
                StateBundle sb = new StateBundle();
                sb.readObject(in);
                componentStates.put(componentName, sb);
        }
        lastModifiedCCSTimeStamp = (CCSTimeStamp) in.readObject();                
    }
}
