package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * A static description of published data.
 * @author LSST CCS Team
 */
public class DataProviderInfo implements Serializable {
    
    private static final long serialVersionUID = -905853368743117573L;

    private final String path;
    private final String key;

    private final Map<Attribute, String> attributes = new LinkedHashMap<>();
        
    public enum Type {
        TRENDING,
        MONITORING,
        CONFIGURATION,
        STATE;
    }
    
    
    /**
     * 
     * @param path The path of the corresponding data provider object
     * @param dataType The type of this data
     * @param name The key of the data. The way the name is used in conjunction
     *             with the path depends on the data type.
     * @param attributes A map of the {@code Attribute}s for this DataProviderInfo
     *              object.
     * The path for all dataTypes is the path of the object the DataProviderInfo
     * applies to. The name depends on the dataType:
     *             CONFIGURATION: 
     *                The name is the name of configuration parameter.
     *             STATE:
     *                the name is the name of the state enumeration.
     *             TRENDING and MONITORING
     *                the name is the trending key
     */
    public DataProviderInfo(String path, Type dataType, String name, Map<Attribute,String> attributes) {
        this.key = name;
        this.path = path;
        if ( attributes != null ) {        
            this.attributes.putAll(attributes);
        }
        this.attributes.put(Attribute.DATA_TYPE, dataType.name());
    }

    /**
     * 
     * @param path The path of the corresponding data provider object
     * @param dataType The type of this data
     * @param name The key of the data. The way the name is used in conjunction
     *             with the path depends on the data type.
     * 
     * The path for all dataTypes is the path of the object the DataProviderInfo
     * applies to. The name depends on the dataType:
     *             CONFIGURATION: 
     *                The name is the name of configuration parameter.
     *             STATE:
     *                the name is the name of the state enumeration.
     *             TRENDING and MONITORING
     *                the name is the trending key
     */
    public DataProviderInfo(String path, Type dataType, String name) {
        this(path, dataType, name, null);
    }

    public void addAttribute(Attribute attribute, String value) {
        if ( value != null && ! value.isEmpty() ) {
            attributes.put(attribute, value);
        }
    }
    
    /**
     * Get access to the different attributes attached to this data description.
     * @param attr the attribute
     * @return 
     */
    public String getAttributeValue(Attribute attr) {
        return attributes.get(attr);
    }

    /**
     * The key of a DataProvider is used to store the data in the trending database.
     * @return  The key for this data.
     */
    public String getKey() {
        return key;
    }

    /**
     * The path of a DataProvider is its location within an Agent.
     * @return  The unique path for this data.
     */
    public String getPath() {
        return path;
    }
    
    public String getFullPath() {
        if ( key.equals(path) ) {
            return path;
        } else {
            return path+"/"+key;
        }
    }

    /**
     * Get all the attribute names defined for this data.
     * @return The array of attribute names.
     */
    public Attribute[] getAttributes() {
        return attributes.keySet().toArray(new Attribute[0]);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Path: ").append(getPath()).append(" key: ").append(getKey()).append("[");
        for ( Attribute a : getAttributes() ) {            
            sb.append(a.getName()).append("=").append(getAttributeValue(a)).append(" ");
        }
        sb.append("]");
        return sb.toString();
    }

    public static enum Attribute {

        DESCRIPTION("description",true),
        UNITS("units",true),
        TYPE("type",true),
        FORMAT("format",true),
        PAGE("page",false),
        SECTION("section",false),
        ALARMHI("alarmhi",false),
        ALARMLO("alarmlo",false),
        STATE("state",true),
        DATA_GROUP("dataGroup",true),
        DATA_TYPE("dataType",false),
        PUBLISHED_PATH("publishedPath",false),
        RTD_TYPE("rtdType",true),
        CONFIG_TYPE("config_type",true),
        CONFIG_CATEGORY("config_category",true),
        CONFIG_RANGE("config_range",true),
        DO_NOT_TREND("don_not_trend",false);
        

        private final String name;
        private final boolean isMetadata;

        private Attribute(String name, boolean isMetadata) {
            this.name = name;
            this.isMetadata = isMetadata;
        }
        
        public String getName() {
            return name;
        }
        
        public boolean isMetadata() {
            return isMetadata;
        }
    }    
        
}
