package org.lsst.ccs.bus.data;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.utilities.structs.ParameterPath;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A Class containing the Agent's configuration information.
 * 
 * @author The LSST CCS Team
 */
public final class ConfigurationInfo implements Serializable { 

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 45966430232923784L;
    
    public static final int DEFAULT_VERSION = -1;
    public static final int LATEST_VERSION = -2;
    public static final int UNDEF_VERSION = -3;
    
    private String descriptionName;
    private String globalName;
    private Integer version = UNDEF_VERSION;
    private final Map<String, Boolean> hasCategoryChanged = new HashMap<>();
    private final Map<String,String> tags = new HashMap<>();
    private final Map<String, Integer> versions = new HashMap<>();
    private ConfigurationState configState;
    private final List<String> recentChanges = new ArrayList<>();
    private final List<ConfigurationParameterInfo> parametersView = new ArrayList<>();
    private CCSTimeStamp ccsTimeStamp;
    
    /**
     * Returns the subsystem description name, ie the name of the description
     * file the subsystem is started with.
     * @return the subsystem description name
     */
    public String getDescriptionName() {
        return descriptionName;
    }
    
    /**
     * Get the CCSTimeStamp of when this object was created.
     * 
     * @return The CCSTimeStamp when this object was created.
     */
    public CCSTimeStamp getCCSTimeStamp() {
        return ccsTimeStamp;
    }

    /**
     * @return the String representation of the configuration state 
     */
    public String getConfigurationName(){
        StringBuilder sb = new StringBuilder(descriptionName);
        if (globalName != null) {
            sb.append("[").append(globalName);
            if(version != null && version != UNDEF_VERSION) {
                sb.append("(").append(version).append(")");
            }
            sb.append("]");
        }
        sb.append("[").append(getTaggedCategories()).append("]");
        return sb.toString();
    }

    public String getGlobalName() {
        return globalName;
    }
    
    /**
     * Returns the String representation of the categories.
     * ex : "catA:A1,catB:B2*,def4"
     * @return the string representation of the categories. 
     */
    private String getTaggedCategories(){
        
        String res = "";
        for (Map.Entry<String,String> entry : new TreeMap<String,String>(tags).entrySet()){
            boolean hasChanges = hasChangesForCategory(entry.getKey());
            Integer v = getConfigVersion(entry.getKey());
            res += (entry.getKey().isEmpty()?"":(entry.getKey()+":"))
                    + entry.getValue()+(v < 0 ? "": "("+v+")")+(hasChanges?"*":"")+",";
        }
        if (!res.isEmpty()){
            res = res.substring(0, res.length()-1);
        }
        return res;
    }

    /**
     * Returns true if there are unsaved changes for the specified category.
     * @param category
     * @return true if there are unsaved changes for the specified category,
     * false otherwise.
     */
    public boolean hasChangesForCategory(String category){
        return hasCategoryChanged.get(category);
    }
    
    /**
     * 
     * @return true if there are no unsaved changes for all categories.
     */
    public boolean hasChanges(){
        for(boolean b : hasCategoryChanged.values()){
            if (b) return true;
        }
        return false;
    }
    
    /**
     * @param category
     * @return the base configuration name for the specified category.
     */
    public String getConfigNameForCategory(String category) {
        return tags.get(category);
    }
    
    /**
     * @param cat
     * @return the version of the configuration applied to the given category.
     */
    public Integer getConfigVersion(String cat) {
        if(versions == null) {
            // For backward compatibility on the ccs buses.
            return UNDEF_VERSION;
        }
        Integer ver = versions.get(cat);
        return ver == null ? UNDEF_VERSION : ver;
    }
    
    /**
     * Tests if {@code category} is one of the subsystem parameter categories.
     * @param category
     * @return true if {@code category} is one of the parameter categories of 
     * the subsystem.
     */
    public boolean hasCategory(String category) {
        return tags.containsKey(category);
    }
    
    /**
     * @return the set of categories the configurable parameters of this {
     * @configurationInfo} are split into.
     */
    public Set<String> getCategorySet() {
        return tags.keySet();
    }
    
    /**
     * @return the configuration state
     */
    public ConfigurationState getConfigurationState(){
        return configState;
    }
    
    /**
     * @return a String representation of the {@ConfigurationInfo} object
     * 
     * DO NOT MODIFY. IT IS USED BY COMMAND getConfigurationState in ConfigurationService.java
     */
    @Override
    public String toString() {
        return "Configuration state : "+ configState + ":" + getConfigurationName();
    }
    
    /**
     * Returns a list of {@code ConfigurationParameterInfo}. Each element of this
     * list represents the current view of a configurable parameter.
     * @return a List of {@code ConfigurationParameterInfo}
     */
    public List<ConfigurationParameterInfo> getAllParameterInfo(){
        return Collections.unmodifiableList(parametersView);
    }
    
    /**
     * Get the list of ConfigurationParameterInfo object whose value has changed
     * between the old and the new ConfigurationInfo.
     * The returned ConfigurationParameterInfo objects belong to the new
     * ConfigurationParameterInfo.
     * 
     * @param ci The ConfigurationInfo object from which we want to calculate the
     *           difference
     * @return   The List of ConfigurationParameterInfo objects that have changed.
     */
    public List<ConfigurationParameterInfo> diff(ConfigurationInfo ci) {
        List<ConfigurationParameterInfo> result = new ArrayList();
        for ( ConfigurationParameterInfo newParInfo : getAllParameterInfo() ) {
            String newParValue = newParInfo.getCurrentValue();
            String oldParValue = ci.getCurrentValueForParameter(newParInfo.getPathName());
            if ( newParValue == null ) {
                if ( oldParValue != null ) {
                    result.add(newParInfo);
                }
            } else if ( !newParValue.equals(oldParValue) ) {
                result.add(newParInfo);
            }
        }
        return result;
    }
    
    /**
     * Returns a list of the last changes that occurred since the previous
     * {@code ConfigurationInfo} publication.
     * <ul>
     * <li>If a single change is processed, the returned list contains this
     * change only.
     * <li>If a bulk change is processed, the returned list contains all the
     * changes that have been submitted, validated and successfully set during
     * this bulk change session.
     * </ul>
     *
     * @return the list of recent changes.
     */
    public List<ConfigurationParameterInfo> getLatestChanges(){
        if ( ! complete ) {
            throw new RuntimeException("Cannot use this ConfigurationInfo object; it's not complete yet.");
        }
        List<ConfigurationParameterInfo> latestChanges = new ArrayList<>();
        for ( String path: recentChanges ) {
            ConfigurationParameterInfo p = configByPath.get(path);
            if ( p != null ) {
                latestChanges.add(p);
            }
        }
        return latestChanges;
    }
    
    // Common actions on the list of parameter info

    /**
     * Returns the current value of each parameter that belong to the specified
     * category.
     * @param category
     * @return a Map of parameter pathName and their current value.
     */
    public Map<String,String> getCurrentValuesForCategory(String category){
        if ( ! complete ) {
            throw new RuntimeException("Cannot use this ConfigurationInfo object; it's not complete yet.");
        }
        return Collections.unmodifiableMap(valuesMapByCategory.get(category));
    }
    
    /**
     * Returns the current value of each parameter that belong to the specified 
     * configurable component.
     * @param componentName the name of the component
     * @return a Map of parameter names to their current value.
     */
    public Map<String, String> getCurrentValuesFor(String componentName) {
        if ( ! complete ) {
            throw new RuntimeException("Cannot use this ConfigurationInfo object; it's not complete yet.");
        }
        return Collections.unmodifiableMap(valuesMapByComponent.get(componentName));
    }
    
    /**
     * Return the current value of a given configurable parameter
     *
     * @param pathName the path name of a parameter, ie
     * {@code moduleName/parameterName}, old format
     * {@code moduleName//parameterName} is still supported but should be
     * replaced)
     * @return a String representation of the parameter's value
     */
    public String getCurrentValueForParameter(String pathName){
        if ( ! complete ) {
            throw new RuntimeException("Cannot use this ConfigurationInfo object; it's not complete yet.");
        }
        if (pathName.contains("//")) {
            System.out.println("double slash should be removed when accessing the value of a parameter by its path");
        }
        ParameterPath pp = ParameterPath.valueOf(pathName);
        return configByPath.get(pp.toString()).getCurrentValue();        
    }


    private Map<String, ConfigurationParameterInfo> configByPath;
    private Map<String, Map<String, String>> valuesMapByCategory;
    private Map<String, Map<String, String>> valuesMapByComponent;
    private boolean complete = false;

    private void complete() {
        complete = true;

        configByPath = new HashMap();
        valuesMapByCategory = new HashMap();
        valuesMapByComponent = new HashMap();
        
        for( ConfigurationParameterInfo par : parametersView ) {
            String pathName = par.getPathName(); 
            String currentValue = par.getCurrentValue();
            String category = par.getCategoryName();
            String component = par.getComponentName();

            configByPath.put(pathName, par);            

            Map<String, String> valuesMapForCategory = valuesMapByCategory.getOrDefault(category, new HashMap<>());
            valuesMapForCategory.put(pathName,currentValue);
            valuesMapByCategory.put(category, valuesMapForCategory);

            Map<String, String> valuesMapForComponent = valuesMapByComponent.getOrDefault(component, new HashMap<>());
            valuesMapForComponent.put(par.getParameterName(),currentValue);
            valuesMapByComponent.put(component, valuesMapForComponent);

        }        
    }
        
    /**
     * Groups the ConfigurationParameterInfo objects of the input list by the component they belong to.
     * @param parmList
     * @return a Map of component name to the list of their parameters.
     */
    public static Map<String, List<ConfigurationParameterInfo>> getParameterInfoGroupByComponent(List<ConfigurationParameterInfo> parmList){
        return parmList.stream()
                .collect(Collectors.groupingBy(ConfigurationParameterInfo::getComponentName, Collectors.toList()));
    }
    
    /**
     * Groups the ConfigurationParameterInfo objects of the input list by the category they belong to.
     * @param parmList
     * @return a Map of category names to the list of parameters that belong to this category.
     */
    public static Map<String, List<ConfigurationParameterInfo>> getParameterInfoGroupByCategory(List<ConfigurationParameterInfo> parmList){
        return parmList.stream()
                .collect(Collectors.groupingBy(ConfigurationParameterInfo::getCategoryName, Collectors.toList()));
    }
    
    
    private void readObject(java.io.ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        if ( ! complete ) {
            complete();
        }
    }
    
    
    /**
     * A builder for a {@ConfigurationInfo} object.
     */
    public static class Builder {
        
        private final ConfigurationInfo ci;
        
        private final Map<ParameterPath, ConfigurationParameterInfo.Builder> parameterBuilders = new HashMap<>();
        
        public Builder() {
            ci = new ConfigurationInfo();
        }
        
        public Builder setDescription(String description) {
            ci.descriptionName = description;
            return this;
        }
        
        public Builder setGlobalConfigurationInformation(String globalName, Integer version) {
            ci.globalName = globalName;
            ci.version = version;
            return this;
        }
        
        public Builder setConfigurationState(ConfigurationState configState) {
            ci.configState = configState;
            return this;
        }
        
        public Builder addParameter(ParameterPath path, Type type, String category, String description, boolean isFinal, boolean isReadOnly) {
            ConfigurationParameterInfo.Builder builder = new ConfigurationParameterInfo.Builder();
            builder.addParameter(path, type, category, description, isFinal, isReadOnly);
            parameterBuilders.put(path, builder);
            ci.parametersView.add(builder.build());
            return this;
        }
        
        public Builder updateParameter(ParameterPath path, String configuredValue, String currentValue, boolean dirty) {
            parameterBuilders.get(path).updateParameter(configuredValue, currentValue, dirty);
            return this;
        }

        public Builder updateCategoryInformation(String cat, String configName, Integer version, boolean dirty) {
            ci.tags.put(cat, configName);
            ci.versions.put(cat, version);
            ci.hasCategoryChanged.put(cat, dirty);
            return this;
        }
        
        public Builder addRecentChange(String path) {
            ci.recentChanges.add(path);
            return this;
        }
        
        public Builder updateVersions(int globalVersion, Map<String, Integer> versions) {
            ci.versions.putAll(versions);
            ci.version = globalVersion;
            return this;
        }
        
        public Builder setCCSTimeStamp(CCSTimeStamp ccsTimeStamp) {
            ci.ccsTimeStamp = ccsTimeStamp;
            return this;
        }

        public ConfigurationInfo build() {
            ci.complete();
            return ci;
        }
        
    }
    
}
