package org.lsst.ccs.bus.messages;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import org.lsst.ccs.bus.data.AgentInfo;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.time.Duration;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Base class for messages sent on the buses.
 * Concrete implementations (command, status and log messages) will be sent
 * on the appropriate bus.
 *
 * Each BusMessage wraps a Serializable Object to be sent over the buses. Of this Object it contains
 * its byte code representation and its Encoded version. The return value of the encoding
 * process depends on the different subclasses. Refer to their documentation for
 * more information.
 * Clients receiving this message can choose to use the encoded version
 * of the embedded Object (which is always available), or, if they have the class
 * definition of the embedded Object, use the de-serialized version of the Object.
 *
 * The embedded object is serialized and encoded when the BusMessage is serialized,
 * i.e. right before being shipped over the buses.
 *
 * Additional information contained by the BusMessage:
 * <ul>
 * <li>class name: the class of the embedded object</li>
 * <li>AgentInfo : the information of the Agent from which the message originated</li>
 * <li>creation timestamp : the timestamp in CCS milliseconds of when the BusMessage was created</li>
 * </ul>
 * This information can be used to filter on the bus messages.
 *
 * @author LSST CCS Team
 * @param <T> Template class for the embedded object. It must be Serializable.
 * @param <D> Template class for the encoded object
 */
public abstract class BusMessage<T extends Serializable, D> implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -51335689636909L;

    private final SerializationUtils<T> serializationUtils = new SerializationUtils<>();

    private D encodedData;
    private byte[] serializedObject;
    /**
     * This field is filled on the receiving side upon invoking the getObject
     * method. It is marked as transient so that no serialization
     * is attempted on it.
     */
    private transient T deserializedObject = null;

    private String className = "";
    private AgentInfo originAgentInfo;

    //Timestamps used to evaluate transfer time
    private volatile transient CCSTimeStamp deserializationTimeStamp;
    private volatile transient CCSTimeStamp doneDeserializationTimeStamp;
    private volatile CCSTimeStamp serializationTimeStamp;
    private volatile transient CCSTimeStamp doneSerializationTimeStamp;
    private final CCSTimeStamp ccsTimeStamp;
    protected final transient T obj;

    /**
     * Build a BusMessage from a the serialized byte array of an object.
     *
     * @param clazz The Class of the serialized object.
     * @param ser The serialized version of the object to be sent over the buses.
     */
    public BusMessage(Class clazz, byte[] ser) {
        this.ccsTimeStamp = CCSTimeStamp.currentTime();
        this.className = clazz.getName();
        this.encodedData = null;
        this.serializedObject = ser;
        this.obj = null;
    }


    /**
     * Build a BusMessage from the provided Object.
     * The provided object can be null, in which case no serialization or encoding occurs.
     *
     * @param obj The Serializable object to be sent over the buses.
     */
    public BusMessage(T obj) {
        this.ccsTimeStamp = CCSTimeStamp.currentTime();
        this.obj = obj;
        if (obj != null) {
            try {
                this.serializedObject = serializationUtils.serialize(obj);
            } catch (IOException ioe) {
                throw new RuntimeException("Could not serialize object.",ioe);
            }
            this.encodedData = encodeObject(obj);
            this.className = obj.getClass().getName();
        } else {
            this.encodedData = null;
            this.serializedObject = null;
        }
    }

    /**
     * Subclasses must provide a specific implementation of the encoding process.
     *
     * @param obj The Serializable object embedded in the BusMessage.
     * @return The encoded version of the embedded object.
     */
    protected abstract D encodeObject(T obj);

    /**
     * Get the class name of the embedded object.
     *
     * @return The class name of the embedded Object.
     *
     */
    public String getClassName() {
        return className;
    }

    /**
     * Get the Encoded version of the embedded Object.
     *
     * @return The Encoded version of the embedded Object.
     */
    public D getEncodedData() {
        return encodedData;
    }

    /**
     * Get the original version of the embedded Object.
     *
     * @return The de serialized version of the embedded Object
     * @throws EmbeddedObjectDeserializationException if the embedded object cannot be de-serialized.
     */
    public T getObject() {
        if ( obj != null ) {
            return obj;
        }
        if (serializedObject == null) {
            return null;
        }
        if (deserializedObject == null) {
            deserializedObject = serializationUtils.deserialize(serializedObject);
        }
        return deserializedObject;
    }

    /**
     * The Origin of the BusMessage, the bus registration name of the Agent from which it
     * originated.
     *
     * @return The origin of this BusMessage.
     */
    public AgentInfo getOriginAgentInfo() {
        return originAgentInfo;
    }

    /**
     * Sets the origin for this BusMessage.
     * This method should only be used internally, by the MessagingLayer Architecture.
     * It must not be used by users/developers.
     *
     * @param agent The origin of the BusMessage.
     *
     */
    public final void setOriginAgentInfo(AgentInfo agent) {
        if (originAgentInfo != null) {
            throw new RuntimeException("The setOriginAgentInfo method on BusMessage must be invoked only once!!!");
        }

        this.originAgentInfo = agent;
    }
    
    /**
     * Get the CCSTimeStamp of when this message was created.
     * 
     * @return The CCSTimeStamp when this object was created.
     */
    public CCSTimeStamp getCCSTimeStamp() {
        return ccsTimeStamp;
    }
    

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(1);
        String sOrigin = getOriginAgentInfo() != null ? getOriginAgentInfo().getName() : "notSet";
        sb.append(this.getClass().getSimpleName()).append(" { origin=").append(sOrigin).append("\n");
        sb.append(" creation timestamp=").append(getCCSTimeStamp()).append("\n");
        if (!className.isEmpty()) {
            sb.append(" className=").append(className).append("\n");
        }
        D ed = getEncodedData();
        if (ed != null) {
            sb.append(" encodedData:\n").append(ed).append("\n");
        }
        return sb.toString();
    }

    /**
     * Method invoked when the object is serialized before being sent the buses.
     * At the time this method is invoked the following will happen:
     * <ul>
     * <li>The embedded object is encoded</li>
     * <li>The embedded object is serialized</li>
     * </ul>
     *
     * @throws IOException
     */
    private void writeObject(ObjectOutputStream out) throws IOException {
        serializationTimeStamp = CCSTimeStamp.currentTime();
        out.defaultWriteObject();
        out.writeObject(CCSTimeStamp.currentTime());
    }

    private void readObject(java.io.ObjectInputStream in) throws IOException, ClassNotFoundException {
        deserializationTimeStamp = CCSTimeStamp.currentTime();
        in.defaultReadObject();
        try {
            doneSerializationTimeStamp = (CCSTimeStamp) in.readObject();
        } catch (Exception e) {
            //Silently ignore exception. 
            //This is for backward compatibility, since we introduced
            //serialization and deserialization timestamps.
            //See: https://jira.slac.stanford.edu/browse/LSSTCCS-1812
//            e.printStackTrace();
        }
        doneDeserializationTimeStamp = CCSTimeStamp.currentTime();
    }
    
    /**
     * Get the Transfer Duration, this is the time it took from when the BusMessage
     * was serialized to when it was de-serialized.
     * @return 
     */
    public Duration getTransferDuration() {
        if ( doneSerializationTimeStamp != null && deserializationTimeStamp != null ) {
            return Duration.between(doneSerializationTimeStamp.getUTCInstant(),deserializationTimeStamp.getUTCInstant());
        }
        return null;
    }
    
    public CCSTimeStamp getDoneDeSerializationTime() {
        return deserializationTimeStamp;        
    }

    public CCSTimeStamp getSerializationTime() {
        return serializationTimeStamp;        
    }

    public Duration getSerializationDuration() {
        if ( serializationTimeStamp != null && doneSerializationTimeStamp != null ) {
            return Duration.between(serializationTimeStamp.getUTCInstant(),doneSerializationTimeStamp.getUTCInstant());
        }
        return null;        
    }
    
    public Duration getDeserializationDuration() {
        if ( deserializationTimeStamp != null && doneDeserializationTimeStamp != null ) {
            return Duration.between(deserializationTimeStamp.getUTCInstant(),doneDeserializationTimeStamp.getUTCInstant());
        }
        return null;        
    }

    /**
     * Private class with Serialization utility methods.
     *
     */
    private class SerializationUtils<T extends Serializable> implements Serializable {

        /**
	 * 
	 */
	private static final long serialVersionUID = -1377509787159869583L;

	/**
         * Serialize the provided object.
         *
         * @param obj The Serializable object to be serialized.
         * @return The byte array serialized representation of the provided object.
         * @throws IOException if the object cannot be serialized.
         *
         */
        public byte[] serialize(T obj) throws IOException {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            ObjectOutputStream oos = new ObjectOutputStream(bos);
            oos.writeObject(obj);
            return bos.toByteArray();
        }

        /**
         * De-serialize the byte array serialized version of an object into the Object itself.
         *
         * @param bytes the byte array serialized version of the object.
         * @return The de-serialized object
         * @throws EmbeddedObjectDeserializationException if the object cannot be de-serialized.
         */
        public T deserialize(byte[] bytes) throws EmbeddedObjectDeserializationException {
            ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
            try {
                ObjectInputStream ois = new ObjectInputStream(bis);
                return (T) ois.readObject();
            } catch (ClassNotFoundException x) {
                throw new EmbeddedObjectDeserializationException("Class not found while deserializing object embedded in message "+this.getClass(), x);
            } catch (IOException x) {
                throw new EmbeddedObjectDeserializationException("Could not deserialize object embedded in message "+this.getClass(), x);
            }
        }

    }
}
