package org.lsst.ccs.bus.messages;

import java.io.Serializable;

/**
 * The result of a CommandRequest.
 * The execution could be either successful or in error. This is reflected by
 * by the content of the returned Object. If in error an exception is returned.
 *
 * @author emarin
 * @param <T> The template for the embedded object.
 */
public final class CommandResult<T extends Serializable> extends CommandReply<T> {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 4323245356909L;

    private final boolean wasSuccessful;
    /**
     * Constructor for a CommandResult.
     * 
     * @param command The original CommandRequest. This is used for the correlation id.
     * @param reply   The result of the execution. It can be null.
     */
    public CommandResult(CommandRequest command, T reply) {
        super(command, reply);
        wasSuccessful = reply == null || ! (reply instanceof Throwable);
    }

    /**
     * Get the result of the CommandRequest execution.
     * If the CommandResultStatus of the execution is:
     *    - OK it will contain the object produced by the execution. It can be null.
     *    - ERROR it will contain the exception produced by the execution. It cannot be null.
     * 
     * @return The actual result of the execution.
     * @throws EmbeddedObjectDeserializationException if the embedded object cannot
     * be deserialized.
     */
    public Object getResult() {
        return getReply();
    }

    /**
     * Was the execution of the CommandRequest successful?
     * @return true/false
     * 
     */
    public boolean wasSuccessful() {
        return wasSuccessful;
    }
    
    @Override
    public String toString() {
        return getClass().getName() + "(" + getResult() + " from "
                + getOriginAgentInfo() + ")";
    }

}
