package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.lsst.ccs.bus.states.AlertState;

/**
 * A summary of the alerts that have been raised by a Subsystem.
 * This class is used internally by a Subsystem to internally keep a record of
 * the Alerts that have been raised.
 * It is also published on the buses to provide a summary of the subsystem's
 * raised alerts.
 * 
 * @author The LSST CCS Team.
 *
 */
public class RaisedAlertSummary implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -9649721904826248L;
    
    // TODO : change to a map at the next run-time incompatible version.
    protected final List<RaisedAlertHistory> raisedAlerts = new ArrayList<>();

    /**
     * Empty constructor.
     */
    public RaisedAlertSummary() {
        
    }
    
    /**
     * Copy constructor for sending over the buses.
     * @param other 
     */
    public RaisedAlertSummary(RaisedAlertSummary other) {
        raisedAlerts.addAll(other.raisedAlerts);
    }
    
    /**
     * Get a RaisedAlert for a given id.
     * 
     * @param alertId The alert Id
     * @return The corresponding RaisedAlert.
     * 
     */
    public RaisedAlertHistory getRaisedAlert(String alertId) {
        for (RaisedAlertHistory raisedAlert : raisedAlerts) {
            if (raisedAlert.getLatestAlert().getAlertId().equals(alertId)) {
                return raisedAlert;
    }
        }
        return null;
    }

    /**
     * Get the overall AlertState for this RaisedAlarm summary object .
     *
     * This is the highest AlertState among all the RaisedAlerts for the
     * subsystem.
     *
     * @return The highest AlertState.
     */
    public AlertState getAlertState() {
        AlertState base = AlertState.NOMINAL;
        for (RaisedAlertHistory alert : raisedAlerts) {
            if (alert.getHighestAlertState().compareTo(base) > 0) {
                base = alert.getHighestAlertState();
            }
        }
        return base;
    }

    /**
     * Get the set of raised Alerts.
     *
     * @return The Set of RaisedAlertHistories.
     */
    public Set<RaisedAlertHistory> getAllRaisedAlertHistories() {
        return new HashSet<>(raisedAlerts);
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Alarm Summary\n");
        synchronized (this) {
            sb.append("Severity : ").append(getAlertState()).append("\n");
            for (RaisedAlertHistory alert : raisedAlerts) {
                sb.append("Alert: ").append(alert.getLatestAlert().getAlertId()).append(" instances: ").append(alert.getNumberOfInstances()).append(" severity: ").append(alert.getHighestAlertState()).append("\n");
                for (RaisedAlertInstance instance : alert.getRaisedAlertInstancesList()) {
                    sb.append("\t").append(instance.getCause()).append(" (").append(instance.getAlertState()).append("@").append(instance.getCCSTimeStamp().getUTCInstant().toEpochMilli()).append(")\n");
                }
            }
        }
        return sb.toString();
    }

}
