package org.lsst.ccs.bus.data;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * This class represents an instance when an Alert was raised.
 * It contains the CCS timestamp of when the Alert was raised and the
 * AlertState with which it was raised. This object can stand for several raised
 * alert events if those alerts are raised at a high rate, in which case the CCS 
 * timestamp corresponds to the first time the alert was raised, and the cause to
 * the latest alert cause.
 * 
 * This class is immutable outside of the package it belongs to.
 * 
 * @author The LSST CCS Team
 * 
 */
public final class RaisedAlertInstance implements Serializable {
    
    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 312950371630569273L;
    
    private final AlertState severity;
    private final long timestamp;
    private CCSTimeStamp ccsTimeStamp;
    private final String description;
    private final int count;
    
    @Deprecated
    RaisedAlertInstance(AlertState severity, long timestamp, String cause) {
        this(severity, timestamp, cause, 1);
    }
    
    RaisedAlertInstance(AlertState severity, CCSTimeStamp ccsTimeStamp, String cause) {
        this(severity, ccsTimeStamp, cause, 1);
    }

    @Deprecated
    RaisedAlertInstance(AlertState severity, long timestamp, String cause, int count) {
        this.severity = severity;
        this.timestamp = timestamp;
        this.description = cause;
        this.count = count;
        this.ccsTimeStamp = CCSTimeStamp.currentTimeFromMillis(timestamp);
    }

    RaisedAlertInstance(AlertState severity, CCSTimeStamp ccsTimeStamp, String cause, int count) {
        this.severity = severity;
        this.timestamp = ccsTimeStamp.getUTCInstant().toEpochMilli();
        this.description = cause;
        this.count = count;
        this.ccsTimeStamp = ccsTimeStamp;
    }

    /**
     * The CCS timestamp of when the RaisedAlertInstance was raised.
     *
     * @return The CCS timestamp.
     * @deprecated uset {@code getCCSTimeStamp()} instead
     */
    public long getTimestamp() {
        return timestamp;
    }

    /**
     * The CCS timestamp of when the RaisedAlertInstance was raised.
     * 
     * @return The CCSTimeStamp.
     */
    public CCSTimeStamp getCCSTimeStamp() {
        return ccsTimeStamp;
    }
   

    /**
     * The AlertState at which the RaisedAlertInstance was raised.
     *
     * @return The AlertState of the Alert.
     */
    public AlertState getAlertState() {
        return severity;
    }
    
    /**
     * What caused this alert to be raised.
     * 
     * @return The description of this instance of the RaisedAlert.
     */
    public String getCause() {
        return description;
    }
        
    /**
     * The number of time this alert was raised.
     * @return 
     */
    public int getCount() {
        return count;
    }
    
    //REMOVE WHEN timestap is removed
    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        if ( this.ccsTimeStamp == null ) {
            ccsTimeStamp = CCSTimeStamp.currentTimeFromMillis(timestamp);
        }
    }
    
}
