package org.lsst.ccs.bus.data;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Generic Data class for Key-Value pairs.
 * This class is meant to be used by CCS Agents to publish key-value pairs type
 * information. Subclasses of this class will provide additional information when
 * needed. For example KeyValueTimeData provides a timestamp for the key-value pair.
 * 
 * @author The LSST CCS Team
 *
 */
public class KeyValueData implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -24918287359238L;
    
    private final String key;
    private final Serializable value;
    private final long timestamp;
    private CCSTimeStamp ccsTimeStamp;

    public static enum KeyValueDataType { KeyValueTrendingData, KeyValuePlotData, KeyValueMetaData };
    private final KeyValueData.KeyValueDataType type;

    
    /**
     * Build a KeyValueData object providing the name and the value for the key-value pair
     * and the timestamp in which this data was accumulated and their type.
     * The type defines what client codes will do with the KeyValueData.
     * 
     * @param key The String representing the key
     * @param value The Object representing the value of the pair.
     * @param timestamp The timestamp is in milliseconds.
     * @deprecated use constructor with CCSTimeStamp
     */
    @Deprecated
    public KeyValueData(String key, Serializable value, long timestamp, KeyValueData.KeyValueDataType type) {        
        this(key, value, CCSTimeStamp.currentTimeFromMillis(timestamp), type);
    }
    public KeyValueData(String key, Serializable value, CCSTimeStamp ccsTimeStamp, KeyValueData.KeyValueDataType type) {
        this.key = key;
        this.value = value;
        this.timestamp = ccsTimeStamp.getUTCInstant().toEpochMilli();
        this.type = type;
        this.ccsTimeStamp = ccsTimeStamp;
    }
    
    /**
     * Build a KeyValueData object providing the name and the value for the key-value pair
     * and the timestamp in which this data was accumulated.
     * 
     * @param key The String representing the key
     * @param value The Object representing the value of the pair.
     * @param timestamp The timestamp is in milliseconds.
     * @deprecated use constructor with CCSTimeStamp
     */
    @Deprecated
    public KeyValueData(String key, Serializable value, long timestamp) {
        this(key,value,timestamp,KeyValueDataType.KeyValueTrendingData);
    }
    public KeyValueData(String key, Serializable value, CCSTimeStamp ccsTimeStamp) {
        this(key,value,ccsTimeStamp,KeyValueDataType.KeyValueTrendingData);
    }

    /**
     * Build a KeyValueData object providing the name and the value for the key-value pair.
     * The timestamp is automatically assigned the time this object was created.
     * 
     * @param key The String representing the key
     * @param value The Object representing the value of the pair.
     */
    public KeyValueData(String key, Serializable value) {
        this(key, value, CCSTimeStamp.currentTime());
    }

    /**
     * Get the timestamp associated with the key-value pair.
     * @return The timestamp in milliseconds.
     * @deprecated uset {@code getCCSTimeStamp()} instead
     */
    public long getTimestamp() {
        return timestamp;
    }            
    
    /**
     * Get the CCSTimeStamp associated with the key-value pair.
     * @return The CCSTimeStamp.
     */
    public CCSTimeStamp getCCSTimeStamp() {
        return ccsTimeStamp;
    }
    
    
    /**
     * Get the value embedding in this object.
     * @return The value.
     */
    public Serializable getValue() {
        return value;
    }

    /**
     * Get the key.
     * @return The name of the Key
     */
    public String getKey() {
        return key;
    }

    /**
     * Get the KeyValueData type.
     * @return The KeyValueData type.
     * 
     */
    public KeyValueDataType getType() {
        return type;
    }
    
    @Override
    public String toString() {
        return "KeyValueData{ key=" + key + ", value=" + value + ", timestamp=" + timestamp+ ", type="+type+" }";
    }
    
    //REMOVE WHEN timestap is removed
    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        if ( this.ccsTimeStamp == null ) {
            ccsTimeStamp = CCSTimeStamp.currentTimeFromMillis(timestamp);
        }
//        this.receivedTimeStamp = System.currentTimeMillis();
    }

}
