package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A history of alerts for a specific alert ID, since the last clear.
 * An instance of this class contains a list of {@link RaisedAlertInstance}
 * objects created by calls to {@code addAlertInstance(...)} method.
 * 
 * This class is meant to be sent over the buses as part of a {@link RaisedAlertSummary} object.
 *
 * @author The LSST CCS Team.
 */
public final class RaisedAlertHistory implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 883621057372902934L;

    private Alert latestAlert;
    private final List<RaisedAlertInstance> instances;
    private AlertState highestSeverity = AlertState.NOMINAL;

    /**
     * The main constructor of the RaisedAlert.
     */
    public RaisedAlertHistory() {
        instances = new ArrayList<>(1);
    }

    /**
     * Get the original Alert that was raised.
     * 
     * @return The original Alert that was raised.
     */
    public Alert getLatestAlert() {
        return latestAlert;
    }
    
    /**
     * Add a new instance of this Alert.
     * 
     * @param severity The AlertState of the Alert.
     * @param alert The Alert that was raised.
     * @param timestamp The CCS timestamp of when the Alert was raised.
     * @param cause The cause of the Alert.
     * @deprecated use the corresponding method with CCSTimeStamp
     */
    @Deprecated
    public void addAlertInstance(AlertState severity, Alert alert, long timestamp, String cause) {
        addAlertInstance(severity, alert, CCSTimeStamp.currentTimeFromMillis(timestamp), cause,1);
    }

    /**
     * Add a new instance of this Alert.
     * 
     * @param severity The AlertState of the Alert.
     * @param alert The Alert that was raised.
     * @param ccsTimeStamp The CCS timestamp of when the Alert was raised.
     * @param cause The cause of the Alert.
     * @deprecated use the corresponding method with CCSTimeStamp
     */
    @Deprecated
    public void addAlertInstance(AlertState severity, Alert alert, CCSTimeStamp ccsTimeStamp, String cause) {
        addAlertInstance(severity, alert, ccsTimeStamp, cause,1);
    }
    
    /**
     * Add a new instance of this Alert.
     * 
     * @param severity The AlertState of the Alert.
     * @param alert The Alert that was raised.
     * @param timestamp The CCS timestamp of when the Alert was raised.
     * @param cause The cause of the Alert.
     * @param count the number of alerts this instance stands for.
     * @deprecated use the corresponding method with CCSTimeStamp
     */
    @Deprecated
    public void addAlertInstance(AlertState severity, Alert alert, long timestamp, String cause, int count) {
        addAlertInstance(severity, alert, CCSTimeStamp.currentTimeFromMillis(timestamp), cause, count);
    }

    /**
     * Add a new instance of this Alert.
     * 
     * @param severity The AlertState of the Alert.
     * @param alert The Alert that was raised.
     * @param ccsTimeStamp The CCS timestamp of when the Alert was raised.
     * @param cause The cause of the Alert.
     * @param count the number of alerts this instance stands for.
     */
    public void addAlertInstance(AlertState severity, Alert alert, CCSTimeStamp ccsTimeStamp, String cause, int count) {
        //REVIEW: This is currently assuming that the instances
        //are already added in the right order.
        //Should we sort this list based on the RaisedAlertInstance?
        if ( severity.compareTo(highestSeverity) > 0 ) {
            highestSeverity = severity;
        }
        instances.add(new RaisedAlertInstance(severity, ccsTimeStamp, cause, count));
        latestAlert = alert;
    }
    
    /**
     * Get the number of times this Alert was raised.
     * 
     * @return the number of times the Alert was raised.
     */
    public int getNumberOfInstances() {
        return instances.size();

    }
    
    /**
     * Get the accumulated number of times this alert was raised.
     * @return the accumulated number of times this alert was raised.
     */
    public int getAccumulatedNumberOfInstances() {
        return instances.stream().collect(Collectors.summingInt(RaisedAlertInstance::getCount));
    }
    
    /**
     * Get the latest AlertState of this Alert.
     * 
     * @return Get the AlertState of the last time this Alert was raised.
     */
    public AlertState getLatestAlertState() {
        int n = getNumberOfInstances();
        switch (n) {
            case 0:
                return AlertState.NOMINAL;
            default:
                return instances.get(n-1).getAlertState();                
        }
    }
    
    /**
     * Get the latest CCS timestamp of this Alert.
     * 
     * @return Get the CCS timestamp of the last time this Alert was raised.
     * @deprecated use {@code getLatestAlertCCSTimeStamp()} instead
     */
    @Deprecated
    public long getLatestAlertTimestamp() {
        CCSTimeStamp ts = getLatestAlertCCSTimeStamp();
        return ts != null ? ts.getUTCInstant().toEpochMilli() : -1;
    }

    /**
     * Get the latest CCS timestamp of this Alert.
     * 
     * @return Get the CCS timestamp of the last time this Alert was raised.
     */
    public CCSTimeStamp getLatestAlertCCSTimeStamp() {
        int n = getNumberOfInstances();
        switch (n) {
            case 0:
                return null;
            default:
                return instances.get(n-1).getCCSTimeStamp();
        }
    }
    
    /**
     * Gets the latest cause of this Alert.
     * @return the cause that was associated to the latest instance of this alert.
     */
    public String getLatestAlertCause() {
        int n = getNumberOfInstances();
        switch (n) {
            case 0:
                return "";
            default:
                return instances.get(n-1).getCause();
        }
    }
    
    /**
     * Gets the latest raised instance of this alert.
     * @return a {@code RaisedAlertInstance} depicting the last time this alert
     * was raised.
     */
    public RaisedAlertInstance getLatestAlertInstance() {
        int n = getNumberOfInstances();
        switch(n) {
            case 0:
                return null;
            default:
                return instances.get(n-1);
        }
    }
    
    /**
     * Get the highest AlertState raised for this Alert.
     * 
     * @return The highest AlertState for this Alert.
     */
    public AlertState getHighestAlertState() {
        return highestSeverity;
    }
    
    /**
     * Get the full history of RaisedAlertInstances for this Alert.
     * 
     * @return List of {@code RaisedAlertInstance}s for this history, in increasing time order.
     */
    public ArrayList<RaisedAlertInstance> getRaisedAlertInstancesList() {
        return new ArrayList<>(instances);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
            sb.append("Raised Alarms for id ").append(getLatestAlert().getAlertId()).append("\n");
            sb.append("Overall Severity: ").append(getHighestAlertState());
        return sb.toString();
    }
    
}
