package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 * A static description of published data.
 * @author LSST CCS Team
 */
public class DataProviderInfo implements Serializable {
    
    private static final long serialVersionUID = -905853368743117573L;

    private final String path;
    private final String key;

    private final Map<Attribute, String> attributes = new HashMap<>();
        
    public enum Type {
        TRENDING,
        MONITORING,
        CONFIGURATION,
        STATE;
    }
    
    public DataProviderInfo(String path, Type dataType, String key, Map<Attribute,String> attributes) {
        this.key = key;
        this.path = path;
        if ( attributes != null ) {        
            this.attributes.putAll(attributes);
        }
        this.attributes.put(Attribute.DATA_TYPE, dataType.name());
    }

    /**
     * 
     * @param path
     * @param key
     * @param attributes
     */
    public DataProviderInfo(String path, String key, Map<Attribute,String> attributes) {
        this.key = key;
        this.path = path;
        if ( attributes != null ) {        
            this.attributes.putAll(attributes);
        }
    }
    
    public DataProviderInfo(String path, Type dataType, String name) {
        this(path, dataType, name, null);
    }

    /**
     * 
     * @param path
     * @param name
     */
    public DataProviderInfo(String path, String name) {
        this(path, name, null);
    }

    public DataProviderInfo(String path, Type dataType) {
        this(path, Type.TRENDING, path, null);
    }
    
    public void addAttribute(Attribute attribute, String value) {
        if ( value != null && ! value.isEmpty() ) {
            attributes.put(attribute, value);
        }
    }
    
    /**
     * Get access to the different attributes attached to this data description.
     * @param attr the attribute
     * @return 
     */
    public String getAttributeValue(Attribute attr) {
        return attributes.get(attr);
    }

    /**
     * The key of a DataProvider is used to store the data in the trending database.
     * @return  The key for this data.
     */
    public String getKey() {
        return key;
    }

    /**
     * The path of a DataProvider is its location within an Agent.
     * @return  The unique path for this data.
     */
    public String getPath() {
        return path;
    }

    /**
     * Get all the attribute names defined for this data.
     * @return The array of attribute names.
     */
    public Attribute[] getAttributes() {
        return attributes.keySet().toArray(new Attribute[0]);
    }


    public static enum Attribute {

        DESCRIPTION("description",true),
        UNITS("units",true),
        TYPE("type",true),
        FORMAT("format",true),
        PAGE("page",false),
        ALARMHI("alarmhi",false),
        ALARMLO("alarmlo",false),
        STATE("state",true),
        RAFT_ID("raftId",true),
        REB_ID("rebId",true),
        AMPLIFIER_ID("amplifierId",true),
        ASPIC_ID("aspicId",true),
        SENSOR_ID("sensorId",true),
        DATA_GROUP("dataGroup",true),
        DATA_TYPE("dataType",false);
        

        private final String name;
        private final boolean isMetadata;

        private Attribute(String name, boolean isMetadata) {
            this.name = name;
            this.isMetadata = isMetadata;
        }
        
        public String getName() {
            return name;
        }
        
        public boolean isMetadata() {
            return isMetadata;
        }
    }    
        
}
