package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import org.apache.log4j.Logger;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.structs.ParameterPath;

/**
 * Represents the state of a given parameter.
 * @author The LSST CCS Team
 */
public class ConfigurationParameterInfo implements Serializable {
    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 85957367834684384L;
    
    private String pathName;
    private String categoryName;
    private String configuredValue;
    private String currentValue;
    private boolean dirty;
    private Boolean isFinal;
    private String componentName;
    private String parameterName;
    @Deprecated
    private String typeName;
    @Deprecated
    private String description;
    @Deprecated
    private String units;
    private Type actualType;
    
    /**
     * The Path name of the parameter.
     * e.g : "moduleName/parameterName"
     * @return a String representation of the path name
     */
    public String getPathName() {
        return pathName;
    }
    
    /**
     * Returns the name of the component the parameter represented by this object
     * belongs to.
     * @return the component name
     */
    public String getComponentName() {
        return componentName;
    }
    
    /**
     * Returns the name of the parameter represented by this object.
     * @return the parameter name
     */
    public String getParameterName() {
        return parameterName;
    }

    /**
     * The category the parameter belongs to.
     * @return the category the parameter represented by this object belongs to
     */
    public String getCategoryName() {
        return categoryName;
    }

    /**
     * The configured value for this parameter, i.e the value defined by the 
     * last configuration loading.
     * @return a String representation of the configured value.
     */
    public String getConfiguredValue() {
        return configuredValue;
    }

    /**
     * The current value for this parameter.
     * @return a String representation of the current value assigned to the parameter
     */
    public String getCurrentValue() {
        return currentValue;
    }

    /**
     * Boolean indicating if this parameter has been changed to a value and has 
     * not been saved yet.
     * @return True if the current value does not match the configured value.
     */
    public boolean isDirty() {
        return dirty;
    }
    
    /**
     * @return The parameter's short description.
     */
    @Deprecated
    public String getDescription() {
        return description;
    }
    
    /**
     * @return The parameter's units
     */
    @Deprecated
    public String getUnits() {
        return units;
    }

    /**
     * Return the type of the parameter.
     * @return 
     */
    @Deprecated
    public String getType() {
        return typeName;
    }
    
    /**
     * Return the actual Type of the parameter
     * @return 
     */
    public Type getActualType() {
        return actualType;
    }
    
    /**
     * Return the current Object value of this parameter
     * @return 
     */
    public Object getCurrentValueObject() {
        return InputConversionEngine.convertArgToType(currentValue, actualType);
    }
    
    /**
     * Return the configured Object value of this parameter
     * @return 
     */
    public Object getConfiguredValueObject() {
        return InputConversionEngine.convertArgToType(configuredValue, actualType);
    }

    public boolean isFinal() {
        if (isFinal == null) {
            Logger.getLogger("org.lsst.ccs.bus.data").warn(
            "this version of ConfigurationParameterInfo does not contain the isFinal information. returning false");
            return false;
        } else {
            return isFinal;
        }
    }

    @Override
    public String toString() {
        return "ConfigurationParameterInfo{" + "pathName=" + pathName + ", categoryName=" + categoryName 
                + ", configuredValue=" + configuredValue 
                + ", currentValue=" + currentValue + ", isDirty=" + dirty + '}';
    }
    
    static class Builder {
        
        private ConfigurationParameterInfo cpi;
        
        Builder() {
            cpi = new ConfigurationParameterInfo();
        }
        
        Builder addParameter(ParameterPath path, Type type, String category, String description, String units, boolean isFinal) {
            cpi.pathName = path.toString();
            cpi.componentName = path.getComponentName();
            cpi.parameterName = path.getParameterName();
            cpi.typeName = type.getTypeName();
            cpi.categoryName = category;
            cpi.description = description;
            cpi.units = units;
            cpi.isFinal = isFinal;
            cpi.actualType = convertTypeIfNeeded(type);
            return this;
        }
        
        Builder updateParameter(String configuredValue, String currentValue, boolean dirty) {
            cpi.configuredValue = configuredValue;
            cpi.currentValue = currentValue;
            cpi.dirty = dirty;
            return this;
        }
        
        ConfigurationParameterInfo build() {
            return cpi;
        }
    }
    
   
    public static class ConfigurationParameterizedType implements ParameterizedType, Serializable {
        
        private final Type[] actualTypeArguments;
        private final Type  rawType;
        private final Type   ownerType;

        ConfigurationParameterizedType(ParameterizedType type) {
            ownerType = convertTypeIfNeeded(type.getOwnerType());
            rawType = convertTypeIfNeeded(type.getRawType());
            int typeIndex = 0;
            actualTypeArguments = new Type[type.getActualTypeArguments().length];
            for ( Type t : type.getActualTypeArguments() ) {
                actualTypeArguments[typeIndex++] = convertTypeIfNeeded(t);
                
            }
        }

        @Override
        public Type[] getActualTypeArguments() {
            return actualTypeArguments;
        }

        @Override
        public Type getOwnerType() {
            return ownerType;
        }

        @Override
        public Type getRawType() {
            return rawType;
        }
        
    }
 
    static Type convertTypeIfNeeded(Type type) {
        if ( type == null ) {
            return null;
        }
        if ( type instanceof Serializable ) {
            return type;
        } else if ( type instanceof ParameterizedType ) {
            return new ConfigurationParameterizedType((ParameterizedType)type);
        } else {
            throw new IllegalArgumentException("Cannot deal with Type: "+type);
        }
    }
    
}
