package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.Objects;

/**
 * Base class for all Alerts.
 * This class provides an alertId field that is used to uniquely identify similar
 * Alert instances: if two Alert instances have the same alertId, they are equal.
 * 
 * Alerts are raised when abnormal situations occur, and they can be raised at
 * a different AlertState level.
 * 
 * Additionally the Alert class has the following field:
 * <ul>
 *  <li>description: the description of the Alert</li>
 * </ul>
 * 
 * Subsystems can either use this directly class or extend it when raising subsystem
 * specific Alerts.
 * When an Alert is raised a StatusAlert message will be published on the Status 
 * bus.
 * 
 * @author The LSST CCS Team
 */
//Implementation note: this class cannot be final as it can be extended by
//subsystem specific Alerts.
public class Alert implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -568951332963342L;

    private final String alertId, description; 
    
    /**
     * Base constructor for all Alerts.
     * 
     * @param alertId The alertId for this Alert instance.
     * @param description The Alert description
     */
    public Alert(String alertId, String description) {
        this.alertId = alertId;
        this.description = description;
    }

    /**
     * Get the Alert id
     * @return The Alert id
     */
    public String getAlertId() {
        return alertId;
    }

    /**
     * Get the Alert description
     * @return The Alert description
     */
    public String getDescription() {
        return description;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 13 * hash + Objects.hashCode(this.alertId);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (! (obj instanceof Alert) ) {
            return false;
        }
        final Alert other = (Alert) obj;
        return Objects.equals(this.alertId, other.alertId);
    }

    @Override
    public String toString() {
        return "Alert "+getAlertId()+" "+getDescription()+")";
    }


    
}
