package org.lsst.ccs.bus.messages;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.bus.annotations.SkipEncoding;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.StateBundle;

/**
 * Base class for all status messages containing data that is meant to be
 * stored in the trending database.
 * The provided Serializable object is embedded within the StatusData object
 * and encoded into the corresponding DataList.
 *
 * @author LSST CCS Team
 */
public class StatusData extends StatusMessage<KeyValueData, KeyValueDataList> {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -12962837209683823L;

    /**
     * Build a StatusMessage from the provided Object.
     *
     * @param obj The Serializable object to be sent over the buses.
     * @param state The State of the Agent at the time the object is created.
     */
    public StatusData(KeyValueData obj, StateBundle state) {
        super(obj, state);
    }

    @Override
    protected KeyValueDataList encodeObject(KeyValueData obj) {
        return EncodingUtils.encodeObject(obj);
    }

    @Override
    public KeyValueDataList getEncodedData() {
        return super.getEncodedData();
    }

    /**
     * Utility class to code/decode objects.
     *
     */
    private static class EncodingUtils {

        private static final Object SKIP_ENCODING = new Object();

        /**
         * An immutable class to describe the Class for a given object.
         * It basically collects the map of fields that can be decoded for a given object.
         * These objects are cached locally for a more efficient encoding.
         *
         */
        private static class ClassDescriptor {

            /**
             * The coding needs the list of fields and their name (when encoding key-value pairs).
             * the decoding needs to link a field to a name (hence the Map).
             * So a LinkedHashMap provides both access through list and map.
             */
            final LinkedHashMap<String, Field> mapFields;

            /**
             * simple constructor for immutable value object
             *
             * @param fieldList
             */
            ClassDescriptor(List<Field> fieldList) {
                if (fieldList != null) {
                    this.mapFields = new LinkedHashMap<>();
                    for (Field field : fieldList) {
                        mapFields.put(field.getName(), field);
                    }
                } else {
                    mapFields = null;
                }
            }

        }

        /**
         * local cache for <TT>ClassDescriptor</TT> objects
         */
        private static final ConcurrentHashMap<Class, ClassDescriptor> classMap = new ConcurrentHashMap<>();

        /**
         * Array of known types that don't need to be encoded.
         *
         */
        private static final Class[] WELL_KNOWN_TYPES = {
            Boolean.class,
            Boolean.TYPE,
            Integer.class,
            Integer.TYPE,
            Float.class,
            Float.TYPE,
            Double.class,
            Double.TYPE,
            Long.class,
            Long.TYPE,
            String.class,
            byte[].class,
            boolean[].class,
            int[].class,
            float[].class,
            double[].class,
            long[].class,
            String[].class,
            // dangerous : Serializable[].class,
            BitSet.class,
            BigDecimal.class,
            List.class,
            Map.class
        };

        /**
         * The NULL_DESCRIPTOR is used to describe objects that are not going to be encoded.
         *
         */
        private static final ClassDescriptor NULL_DESCRIPTOR = new ClassDescriptor(null) {
            @Override
            public String toString() {
                return "NULL_DESCRIPTOR";
            }
        };

        /**
         * The ENUM_DESCRIPTOR is specific to all enums.
         * Enums will be encoded in a special way: the key-value list will have an empty key and a String value
         * that represents the enums instance.
         */
        private static final ClassDescriptor ENUM_DESCRIPTOR = new ClassDescriptor(null) {
            @Override
            public String toString() {
                return "ENUM_DESCRIPTOR";
            }
        };

        /**
         * All the WELL_KNOWN_TYPES are assigned the NULL_DESCRIPTOR so that they
         * will not be encoded.
         *
         */
        static {
            for (Class clazz : WELL_KNOWN_TYPES) {
                classMap.put(clazz, NULL_DESCRIPTOR);
            }
        }

        /**
         * gets a <TT>ClassDescriptor</TT> object for a given class.
         * The descriptor cache is read and if no descriptor is found a new one is built
         * and inserted into the cache.
         *
         * @param clazz The class of the Object for which we are getting the descriptor.
         * @return The ClassDescriptor for the provided Class.
         *
         */
        private static ClassDescriptor getDescriptorFor(Class clazz) {
            if (Map.class.isAssignableFrom(clazz)) {
                return NULL_DESCRIPTOR;
            }
            ClassDescriptor first = classMap.get(clazz);
            if (first == null) {
                first = buildDescriptorFor(clazz);
                classMap.put(clazz, first);
            }
            return first;
        }

        /**
         * Builds a <TT>ClassDescriptor</TT>a for a given class if it was not already built.
         *
         * @param clazz The Class for which to build the ClassDescriptor.
         * @return The ClassDescriptor for the provided class.
         *
         */
        private static synchronized ClassDescriptor buildDescriptorFor(Class clazz) {
            if (clazz.isEnum()) {
                return ENUM_DESCRIPTOR;
            }
            ArrayList<Field> listFields = new ArrayList<>();
            //Get the superclass and find all the inherited fields.
            Class superClazz = clazz.getSuperclass();
            if (superClazz != null) {
                if (!Object.class.equals(superClazz)) {
                    ClassDescriptor superDescriptor = getDescriptorFor(superClazz);
                    if (superDescriptor != NULL_DESCRIPTOR && superDescriptor.mapFields != null) {
                        listFields.addAll(superDescriptor.mapFields.values());
                    }
                }
            }
            Field[] fields = clazz.getDeclaredFields();
            for (Field field : fields) {
                int modifiers = field.getModifiers();
                if (Modifier.isStatic(modifiers)) {
                    continue;
                }
                /*
                 THis code has been added because some classes (such as ArrayList) use transient only for implementation reasons.
                 so the transient field cannot be used for encoding/decoding ....  this is to circumvent this feature...
                 */
                // if transient and of complex type: skip? examples : Thread, IO ... and so on (transient and not Serializable)
                if (Modifier.isTransient(modifiers)) {
                    if (!Serializable.class.isAssignableFrom(field.getType())) {
                        continue;
                    }
                }
                if (field.getAnnotation(SkipEncoding.class) != null) {
                    continue;
                }
                field.setAccessible(true);
                listFields.add(field);
            }
            ClassDescriptor res = new ClassDescriptor(listFields);
            return res;
        }

        /**
         * Encodes an object and return the DataList object containing all the KeyValueData objects
         * in which this Object has been decomposed.
         *
         * @param value The Object to be decomposed.
         * @return The KeyValueDataList containing the KeyValueData in which this Object has been decomposed.
         */
        public static final KeyValueDataList encodeObject(KeyValueData value) {
            String key = "";
            long t = value.getTimestamp();
            Object obj = innerEncodeObject(value, key, t);
            if (obj instanceof KeyValueDataList) {
                return (KeyValueDataList) obj;
            }
            if (obj == null) {
                return null;
            }
            KeyValueDataList d = new KeyValueDataList();
            addObjectToDataList(key, obj, d, t, KeyValueData.KeyValueDataType.KeyValueTrendingData);
            return d;
        }

        protected static final Object innerEncodeObject(Object value, String key, long timestamp) {
            if (value == null) {
                return null;
            }
            Class clazz = value.getClass();

            if (clazz.getAnnotation(SkipEncoding.class) != null) {
                return SKIP_ENCODING;
            }

            ClassDescriptor descriptor = getDescriptorFor(clazz);

            if (value instanceof Collection) {
                Collection c = (Collection) value;
                KeyValueDataList dataList = new KeyValueDataList();
                for (Object val : c) {
                    Object o = innerEncodeObject(val, key, timestamp);
                    if (o != SKIP_ENCODING) {
                        addObjectToDataList(key, o, dataList, timestamp, KeyValueData.KeyValueDataType.KeyValueTrendingData);
                    }
                }
                return dataList;
            }

            boolean isMap = value instanceof Map;
            if (isMap) {
                Map map = (Map) value;
                KeyValueDataList dataList = new KeyValueDataList();
                for (Object obj : map.keySet()) {
                    String name = updateName(obj.toString(), key);
                    Object o = innerEncodeObject(map.get(obj), name, timestamp);
                    if (o != SKIP_ENCODING) {
                        addObjectToDataList(name, o, dataList, timestamp, KeyValueData.KeyValueDataType.KeyValueTrendingData);
                    }
                }
                return dataList;
            }
            
            if (value instanceof BitSet) {
                BitSet bitset = (BitSet) value;
                if ( bitset.length() > 64 ) {
                    throw new RuntimeException("Cannot encode a BitSet with more than 64 bits");
                }
                long l = 0L;
                for (int i = 0; i < bitset.length(); ++i) {
                    l += bitset.get(i) ? (1L << i) : 0L;
                }
                return l;
            }

            if (descriptor == NULL_DESCRIPTOR) {
                return value;
            }
            if (descriptor == ENUM_DESCRIPTOR) {
                return String.valueOf(value);
            }

            if (value instanceof KeyValueData) {
                KeyValueDataList res = new KeyValueDataList();
                KeyValueData kvd = ((KeyValueData) value);
                timestamp = kvd.getTimestamp();
                String name = updateName(kvd.getKey(), key);
                Object o = innerEncodeObject(kvd.getValue(), name, timestamp);
                if (o != SKIP_ENCODING) {
                    addObjectToDataList(name, o, res, timestamp, kvd.getType());
                }
                return res;
            }
            Collection<Field> descriptorValues = descriptor.mapFields.values();
            KeyValueDataList res = new KeyValueDataList();
            if (value instanceof Exception) {
                ((Exception) value).printStackTrace();
            }
            for (Field field : descriptorValues) {
                try {
                    String name = updateName(field.getName(), key);
                    Object val = innerEncodeObject(field.get(value), name, timestamp);
                    if (val != SKIP_ENCODING) {
                        addObjectToDataList(name, val, res, timestamp, KeyValueData.KeyValueDataType.KeyValueTrendingData);
                    }
                } catch (IllegalArgumentException | IllegalAccessException e) {
                    throw new RuntimeException("Something went really wrong encoding the object.", e);
                }
            }
            return res;
        }

        private static String updateName(String name, String prefix) {
            String tmpName = name;
            if (prefix != null && !prefix.isEmpty()) {
                tmpName = prefix + "/" + tmpName;
            }
            return tmpName;
        }

        private static void addObjectToDataList(String name, Object val, KeyValueDataList res, long timestamp, KeyValueData.KeyValueDataType type) {
            if (val instanceof KeyValueDataList) {
                for (KeyValueData d : (KeyValueDataList) val) {
                    res.addData(d);
                }
            } else if (val instanceof KeyValueData) {
                res.addData((KeyValueData) val);
            } else {
                res.addData(name, (Serializable) val, timestamp, type);
            }
        }

    }

}
