package org.lsst.ccs.bus.states;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.utilities.time.CCSInstantFactory;

/**
 * Contains a set of subsystem states, each represented by {@code Enum}. 
 * States are stored internally in a {@code ConcurrentHashMap}, mapped by their class names.
 *
 * This class is meant to be published on the buses with every status message
 * and when there is any state change.
 * 
 * This class is thread safe to the extent supported by the underlying {@code ConcurrentHashMap}:
 * no concurrent modifications are thrown, but atomicity of operations that involve
 * reading or modifying more than one elemental state is not guaranteed.
 *
 * @author The LSST CCS Team
 */
public class StateBundle implements Serializable, Cloneable {

    private static final long serialVersionUID = -5193848741969746240L;
    
    private Map<String, Enum> allStates = new ConcurrentHashMap<>();

    /** 
     * Should remain empty on the sending side. It is filled if needed during
     * deserialization. It does not participate in any logic made on the 
     * StateBundle object (diff or merge).
     */
    private transient Map<String, String> subsystemSpecificStates = new HashMap<>();
    
    private Instant lastModified = CCSInstantFactory.now();

    /** Constructs a StateBundle from a set of states.
     * @param states The set of States to construct the StateBundle with.
     */
    public StateBundle(Enum... states) {
        for (Enum state : states) {
            innerSetState(state);
        }
        updateLastModified();
    }

    /** Copy constructor. */
    private StateBundle(StateBundle b) {
        this.allStates.putAll(b.allStates);
        this.subsystemSpecificStates.putAll(b.subsystemSpecificStates);
        this.lastModified = b.lastModified;
    }

    /**
     * Merge the content of two StateBundle objects returning an updated cloned
     * version of the original StateBundle object.
     * The original StateBundle is left unchanged.
     * 
     * @param newState The StateBundle with the changes.
     * @return An update cloned version of the original StateBundle.
     *
     */
    public StateBundle mergeState(StateBundle newState) {
        StateBundle clone = this.clone();
        clone.allStates.putAll(newState.allStates);
        clone.updateLastModified();
        return clone;
    }

    /**
     * Finds what has changed.
     * 
     * @param oldState The StateBundle of the previous state .
     * @return a new StateBundle containing only the changed states.
     */
    public StateBundle diffState(StateBundle oldState) {
        StateBundle diff = new StateBundle();
        for (Enum e : allStates.values()) {
            if (!oldState.isInState(e)) {
                diff.innerSetState(e);
            }
        }
        diff.updateLastModified();
        return diff;
    }

    /**
     * Sets the specified state to the bundle, replacing any previous state of the same type.
     * @param state The state to set on the StateBundle.
     */
    public final void setState(Enum state) {
        innerSetState(state);
        updateLastModified();
    }

    /**
     * Sets specified states to the bundle, replacing any previous states of the same types. 
     * @param states The specified states to set on the StateBundle
     */
    public final void setState(Enum... states) {
        for (Enum state : states) innerSetState(state);
        updateLastModified();
    }
    
    //Sets the state without updating the timestamp of the last update.
    private void innerSetState(Enum state) {
        allStates.put(state.getClass().getSimpleName(), state);
    }
    
    /**
     * Returns the current value for the specified State.
     *
     * @param <T> An {@code Enum} that defines a set of states.
     * @param enumClass The class of the {@code Enum} representing the state.
     * @return The current value for the given state.
     */
    public final <T extends Enum<T>> Enum getState(Class<T> enumClass) {
        return allStates.get(enumClass.getSimpleName());
    }
    
    /**
     * Check if this {@code StateBundle} contains the specified state.
     *
     * @param <T> An {@code Enum} that defines a set of states.
     * @param state The value of the {@code Enum} to check.
     * @return true if this {@code StateBundle} contains the provided state
     */
    public <T extends Enum<T>> boolean isInState(T state) {
        Enum e = getState(state.getClass());
        return e == null ? false : e.equals(state);
    }

    /**
     * Check if this StateBundle is in all the states of a given StateBundle.
     * 
     * @param stateBundle The StateBundle containing all the states to check.
     * @return true if this StateBundle is in all the states of the provided one.
     */
    public boolean isInState(StateBundle stateBundle) {
        for (Enum e : stateBundle.allStates.values()) {
            if (!isInState(e)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Get the {@code Instant} of the last time the StateBundle
     * was modified.
     * @return The {@code Instant} of the last change
     */
    public Instant getLastModified() {
        return lastModified;
    }
    
    /**
     * Get the Map of all the states that were decoded when this message
     * was received, i.e. all the states for which the client had the
     * class definition. All the other states can be fetched via the getInternalStates() method.
     * 
     * @return The map of all the decoded states.
     */
    public Map<String,Enum> getDecodedStates() {
        return (Map<String,Enum>)Collections.unmodifiableMap(allStates);
    }
    
    /**
     * Get the Map of all the states in this StateBundle.
     * Since it is not guaranteed that all states were decoded upon receipt 
     * we convert all the state Enums to Strings.
     * 
     * @return The map of all the states as strings.
     */
    public Map<String,String> getAllStatesAsStrings() {
        Map<String,String> allStatesAsStrings = new HashMap<>();
        for ( Entry<String,Enum> entry : allStates.entrySet() ) {
            allStatesAsStrings.put(entry.getKey(), entry.getValue().toString());
        }
        allStatesAsStrings.putAll(subsystemSpecificStates);
        return allStatesAsStrings;
    }

    /**
     * Gets string representations of the internal states.
     * @return a map of enum class name to the string value of this enum.
     */
    public Map<String, String> getInternalStates() {
        return Collections.unmodifiableMap(subsystemSpecificStates);
    }
    
    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof StateBundle)) {
            return false;
        }
        StateBundle state = (StateBundle) o;
        if ( allStates.size() != state.allStates.size() ) {
            return false;
        }
        for (String c : allStates.keySet()) {
            if (!state.isInState(allStates.get(c))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode() {
        int result = 0;
        for (Enum e : allStates.values()) {
            result = 31 * result + e.hashCode();
        }
        return result;
    }

    @Override
    public StateBundle clone() {
        StateBundle b = new StateBundle(this);
        return b;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        allStates.forEach((k,v) -> sb.append(k).append(":").append(v).append(","));
        return sb.length() > 0 ? sb.substring(0, sb.length() - 1) : "Empty state";
    }

    private void updateLastModified() {        
        lastModified = CCSInstantFactory.now();        
    }
    
    /**
     * all the states (framework and subsystem specific) are written in the same
     * map.
     */
    private void writeObject(ObjectOutputStream out) throws IOException {
        out.writeInt(allStates.size()+subsystemSpecificStates.size());
        for (Enum e : allStates.values()) {
            out.writeObject(e.getClass().getName());
            out.writeObject(e.toString());
        }
        for (Map.Entry<String, String> entry : subsystemSpecificStates.entrySet()) {
            out.writeObject(entry.getKey());
            out.writeObject(entry.getValue());
        }
        out.writeObject(getLastModified());
    }

    /** For enums the class of which is locally unknown, their class name and value
     are stored in the subsystemSpecificStates map. */
    private void readObject(ObjectInputStream in) throws IOException,
            ClassNotFoundException {
        int n = in.readInt();
        allStates = new ConcurrentHashMap<String, Enum>();
        subsystemSpecificStates = new ConcurrentHashMap<>();
        for (int i = 0; i < n; i++) {
            String className = (String) in.readObject();
            String value = (String) in.readObject();

            try {
                Class<Enum> c = (Class<Enum>) Class.forName(className);
                Method m = c.getDeclaredMethod("valueOf", String.class);
                Enum e = (Enum) m.invoke(null, value);
                allStates.put(c.getSimpleName(), e);
            } catch (Exception e) {
                // Store as a pair of String in subsystemSpecificStates.
                subsystemSpecificStates.put(className, value);
            }

        }
        lastModified = (Instant)in.readObject();
    }
}
