package org.lsst.ccs.bus.messages;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import org.lsst.ccs.bus.data.AgentInfo;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

/**
 * Base class for messages sent on the buses.
 * Concrete implementations (command, status and log messages) will be sent
 * on the appropriate bus.
 *
 * Each BusMessage wraps a Serializable Object to be sent over the buses. Of this Object it contains
 * its byte code representation and its Encoded version. The return value of the encoding
 * process depends on the different subclasses. Refer to their documentation for
 * more information.
 * Clients receiving this message can choose to use the encoded version
 * of the embedded Object (which is always available), or, if they have the class
 * definition of the embedded Object, use the de-serialized version of the Object.
 *
 * The embedded object is serialized and encoded when the BusMessage is serialized,
 * i.e. right before being shipped over the buses.
 *
 * Additional information contained by the BusMessage:
 * <ul>
 * <li>class name: the class of the embedded object</li>
 * <li>AgentInfo : the information of the Agent from which the message originated</li>
 * <li>creation timestamp : the timestamp in CCS milliseconds of when the BusMessage was created</li>
 * </ul>
 * This information can be used to filter on the bus messages.
 *
 * @author LSST CCS Team
 * @param <T> Template class for the embedded object. It must be Serializable.
 * @param <D> Template class for the encoded object
 */
public abstract class BusMessage<T extends Serializable, D> implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 345689513329636909L;

    private final SerializationUtils<T> serializationUtils = new SerializationUtils<>();

    private D encodedData;
    private byte[] serializedObject;
    /**
     * This field is filled on the receiving side upon invoking the getObject
     * method. It is marked as transient so that no serialization
     * is attempted on it.
     */
    private transient T deserializedObject = null;
    /**
     * This field is filled on the sender side only. It is marked as transient
     * so that it does not get serialized.
     */
    private final transient T embeddedObject;
    private String className = "";
    private AgentInfo originAgentInfo;
    //Timestamp in milliseconds when the objects is sent over the buses.
    private final long timeStamp;

    /**
     * Build a BusMessage from the provided Object.
     * The provided object can be null, in which case no serialization or encoding occurs.
     *
     * @param obj The Serializable object to be sent over the buses.
     */
    public BusMessage(T obj) {
        this.timeStamp = System.currentTimeMillis();
        this.embeddedObject = obj;
        if (obj != null) {
            this.className = obj.getClass().getName();
        } else {
            this.encodedData = null;
            this.serializedObject = null;
        }
    }

    /**
     * Subclasses must provide a specific implementation of the encoding process.
     *
     * @param obj The Serializable object embedded in the BusMessage.
     * @return The encoded version of the embedded object.
     */
    protected abstract D encodeObject(T obj);

    /**
     * Get the class name of the embedded object.
     *
     * @return The class name of the embedded Object.
     *
     */
    public String getClassName() {
        return className;
    }

    /**
     * Get the Encoded version of the embedded Object.
     *
     * @return The Encoded version of the embedded Object.
     */
    public D getEncodedData() {
        //Check if we are on the sender side, if so, do the encoding on the fly.
        if (embeddedObject != null) {
            return encodeObject(embeddedObject);
        }
        return encodedData;
    }

    /**
     * Get the original version of the embedded Object.
     *
     * @return The de serialized version of the embedded Object
     * @throws RuntimeException if the embedded object cannot be de-serialized.
     */
    public T getObject() {
        if (embeddedObject != null) {
            return embeddedObject;
        }
        if (serializedObject == null) {
            return null;
        }
        if (deserializedObject == null) {
            try {
                deserializedObject = serializationUtils.deserialize(serializedObject);
            } catch (IOException e) {
                throw new RuntimeException("Problem de-serializing the embedded object.", e);
            }
        }
        return deserializedObject;
    }

    /**
     * The Origin of the BusMessage, the bus registration name of the Agent from which it
     * originated.
     *
     * @return The origin of this BusMessage.
     */
    public AgentInfo getOriginAgentInfo() {
        return originAgentInfo;
    }

    /**
     * Sets the origin for this BusMessage.
     * This method should only be used internally, by the MessagingLayer Architecture.
     * It must not be used by users/developers.
     *
     * @param agent The origin of the BusMessage.
     *
     */
    public final void setOriginAgentInfo(AgentInfo agent) {
        if (originAgentInfo != null) {
            throw new RuntimeException("The setOriginAgentInfo method on BusMessage must be invoked only once!!!");
        }

        this.originAgentInfo = agent;
    }

    /**
     * Get the timestamp in CCS milliseconds of when the message was created.
     *
     * @return The creation time in CCS milliseconds.
     */
    public long getTimeStamp() {
        return timeStamp;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        String sOrigin = getOriginAgentInfo() != null ? getOriginAgentInfo().getName() : "notSet";
        sb.append(this.getClass().getSimpleName()).append(" { origin=").append(sOrigin).append("\n");
        sb.append(" creation timestamp=").append(getTimeStamp()).append("\n");
        sb.append(" className=").append(className).append("\n");
        sb.append(" encodedData: \n");
        sb.append(getEncodedData());
        return sb.toString();

    }

    /**
     * Method invoked when the object is serialized before being sent the buses.
     * At the time this method is invoked the following will happen:
     * <ul>
     * <li>The embedded object is encoded</li>
     * <li>The embedded object is serialized</li>
     * </ul>
     *
     * @throws IOException
     */
    private void prepareForSendingOverTheBuses() throws IOException {
        if (embeddedObject != null) {
            this.encodedData = encodeObject(embeddedObject);
            this.serializedObject = serializationUtils.serialize(embeddedObject);
        }
//        sentTimeStamp = System.currentTimeMillis();
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        prepareForSendingOverTheBuses();
        out.defaultWriteObject();
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
//        this.receivedTimeStamp = System.currentTimeMillis();
    }

    /**
     * Private class with Serialization utility methods.
     *
     */
    private class SerializationUtils<T extends Serializable> implements Serializable {

        /**
	 * 
	 */
	private static final long serialVersionUID = -1377509787159869583L;

	/**
         * Serialize the provided object.
         *
         * @param obj The Serializable object to be serialized.
         * @return The byte array serialized representation of the provided object.
         * @throws IOException if the object cannot be serialized.
         *
         */
        public byte[] serialize(T obj) throws IOException {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            ObjectOutputStream oos = new ObjectOutputStream(bos);
            oos.writeObject(obj);
            return bos.toByteArray();
        }

        /**
         * De-serialize the byte array serialized version of an object into the Object itself.
         *
         * @param bytes the byte array serialized version of the object.
         * @return The de-serialized object
         * @throws Exception if the object cannot be de-serialized.
         */
        public T deserialize(byte[] bytes) throws IOException {
            ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
            ObjectInputStream ois = new ObjectInputStream(bis);
            try {
                return (T) ois.readObject();
            } catch (ClassNotFoundException x) {
                throw new IOException("Class not found while deserializing embedded object", x);
            }
        }

    }
}
