package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.bus.states.AlertState;

/**
 * A history of alerts for a specific alert ID.
 * An instance of this class contains a list of {@link RaisedAlertInstance}
 * objects created by calls to {@code addAlertInstance(...)} method.
 * 
 * This class is meant to be sent over the buses as part of a {@link RaisedAlertSummary} object.
 * The class is thread-safe as long as its instances are safely published.
 *
 * @author The LSST CCS Team.
 */
public final class RaisedAlertHistory implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 883621057372902934L;

    private Alert latestAlert;
    private final List<RaisedAlertInstance> instances;
    private AlertState highestSeverity = AlertState.NOMINAL;

    /**
     * The main constructor of the RaisedAlert.
     */
    public RaisedAlertHistory() {
        instances = new ArrayList<>(1);
    }
    
    /**
     * Copy constructor.
     * @param other Instance to be cloned.
     */
    public RaisedAlertHistory(RaisedAlertHistory other) {
        synchronized (other) {
            latestAlert = other.latestAlert;
            highestSeverity = other.highestSeverity;
            instances = new ArrayList<>(other.instances);
        }
    }

    /**
     * Get the original Alert that was raised.
     * 
     * @return The original Alert that was raised.
     */
    synchronized public Alert getLatestAlert() {
       return latestAlert;
    }
    
    /**
     * Add a new instance of this Alert.
     * 
     * @param severity The AlertState of the Alert.
     * @param alert The Alert that was raised.
     * @param timestamp The CCS timestamp of when the Alert was raised.
     * @param cause The cause of the Alert.
     */
    synchronized public void addAlertInstance(AlertState severity, Alert alert, long timestamp, String cause) {
        //REVIEW: This is currently assuming that the instances
        //are already added in the right order.
        //Should we sort this list based on the RaisedAlertInstance?
        if ( severity.compareTo(highestSeverity) > 0 ) {
            highestSeverity = severity;
        }
        instances.add(new RaisedAlertInstance(severity, timestamp, cause));
        latestAlert = new Alert(alert.getAlertId(), alert.getDescription());
    }

    /**
     * Add a new instance of this Alert.
     * The current CCS timestamp is used.
     * 
     * @param severity The AlertState of the Alert.
     * @param alert The Alert that was raised.
     */
    protected void addAlertInstance(AlertState severity, Alert alert) {
        addAlertInstance(severity,alert, System.currentTimeMillis(),alert.getDescription());
    }
    
    /**
     * Get the number of times this Alert was raised.
     * 
     * @return the number of times the Alert was raised.
     */
    synchronized public int getNumberOfInstances() {
        return instances.size();
    }
    
    /**
     * Get the latest AlertState of this Alert.
     * 
     * @return Get the AlertState of the last time this Alert was raised.
     */
    synchronized public AlertState getLatestAlertState() {
        int n = getNumberOfInstances();
        switch (n) {
            case 0:
                return AlertState.NOMINAL;
            default:
                return instances.get(n-1).getAlertState();                
        }
    }
    
    /**
     * Get the latest CCS timestamp of this Alert.
     * 
     * @return Get the CCS timestamp of the last time this Alert was raised.
     */
    synchronized public long getLatestAlertTimestamp() {
        int n = getNumberOfInstances();
        switch (n) {
            case 0:
                return -1;
            default:
                return instances.get(n-1).getTimestamp();
        }
    }
    
    /**
     * Get the highest AlertState raised for this Alert.
     * 
     * @return The highest AlertState for this Alert.
     */
    synchronized public AlertState getHighestAlertState() {
        return highestSeverity;
    }
    
    /**
     * Get the full history of RaisedAlertInstances for this Alert.
     * 
     * @return The List containing the RaisedAlertInstances for this Alert.
     */
    synchronized public List<RaisedAlertInstance> getRaisedAlertInstancesList() {
        return new ArrayList<>(instances);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        synchronized (this) {
            sb.append("Raised Alarms for id ").append(getLatestAlert().getAlertId()).append("\n");
            sb.append("Overall Severity: ").append(getHighestAlertState());
        }
        return sb.toString();
    }
    
}
