package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.Objects;

/**
 * A class to encapsulate an agent's name and type.
 * This class is used to represent an Agent across the buses.
 * The Agent's name is its unique name on the Buses.
 * Its type defines its role on the buses.
 * 
 * Two AgentInfo instances are considered equal if they have the same name and type.
 * When two AgentInfo instances are equal, they represent the same Agent on
 * the buses.
 *
 * @author emarin
 */
//REVIEW: Since the name is guaranteed to be unique, it seems not necessary
//to test the type for equality.
public final class AgentInfo implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 716901596341446997L;

    private final String name;
    private final AgentType type;

    /**
     * The base constructor of an AgentInfo. It takes the Agent's name and its 
     * AgentType.
     * 
     * @param name The name of the Agent as defined on the Buses.
     * @param type The AgentType of the Agent. This determines its role.
     */
    public AgentInfo(String name, AgentType type) {
        this.name = name;
        this.type = type;
    }

    /**
     * The Agent's name on the Buses. This must be unique.
     * 
     * @return The name of the Agent on the Buses.
     */
    public String getName() {
        return name;
    }

    /**
     * The Agent's AgentType. This represents its role on the buses.
     * 
     * @return The AgentType of the Agent.
     */
    public AgentType getType() {
        return type;
    }

    /**
     * Enumeration of known agent types.
     * 
     */
    public static enum AgentType {

        LISTENER("Listener"),
        CONSOLE("Console"),
        WORKER("Worker"),
        SERVICE("Service"),
        MCM("MCM"),
        OCS_BRIDGE("OCS bridge"),
        LOCK_MANAGER("Lock manager");

        private final String displayName;

        private AgentType(String displayName) {
            this.displayName = displayName;
        }

        /**
         * Get the name of the AgentType for display purposes.
         * 
         * @return The display name of the AgentType.
         */
        //REVIEW: this method seems to be for GUI display purposes.
        //Maybe it does not belong here.
        public String displayName() {
            return displayName;
        }
    }

    @Override
    public String toString() {
        return name + ":" + type;
    }

    @Override
    public boolean equals(Object a) {
        if (!(a instanceof AgentInfo)) {
            return false;
        } else {
            AgentInfo agent = (AgentInfo) a;
            return getName().equals(agent.getName()) && getType().equals(agent.getType());
        }

    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 89 * hash + Objects.hashCode(this.name);
        hash = 89 * hash + Objects.hashCode(this.type);
        return hash;
    }

}
