package org.lsst.ccs.bus.messages;

import java.io.Serializable;
import java.util.UUID;

/**
 * Superclass for all messages sent on the Command Bus.
 * This class contains the following additional fields:
 * <ul>
 *   <li>destination: the destination on the buses of the CommandMessage</li>
 *   <li>correlationId: the correlation id for this message. This is used internally
 *                      to match requests and replies. Each CommandRequest will 
 *                      generate a new unique id which will be used by all CommandReply
 *                      objects related to the CommandRequest.</li>
 * </ul>
 * 
 * The embedded object of a CommandMessage depends on the specific subclass.
 * CommandRequests contain the BasicCommand object that initiated the request.
 * CommandReply objects can either contain an object produced by the execution
 * of the CommandRequest or an Throwable if the execution is not successful.
 * 
 * @author LSST CCS Team
 * @param <T> The Serializable object embedded in the message.
 * @param <D> The type of object returned by the encoding
 */
public abstract class CommandMessage<T extends Serializable ,D> extends BusMessage<T,D> {
        
    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 56321332943534L;

    private final String destination;
    private final UUID correlationId;
        
    /**
     * Default CommandMessage constructor.
     * @param destination The destination of the CommandMessage.
     * @param correlationId The correlation Id for this command.
     * @param obj           The object contained in the CommandMessage.
     */
    protected CommandMessage(String destination, UUID correlationId, T obj) {
        super(obj);
        this.destination = destination;
        this.correlationId = correlationId;                
    }
    
    
    /**
     * The destination of the CommandMessage.
     * This is the name of the CCS Agent on the buses that is the target of this
     * CommandMessage.
     * 
     * @return The destination for this CommandMessage
     */
    public String getDestination(){
        return destination;
    }
    
    /**
     * The correlationId of this CommandMessage. The correlationId is a unique
     * identifier used to correlate CommandMessages related to the same command
     * exchange, as described in the Command State Diagram:
     * https://confluence.slac.stanford.edu/display/LSSTCAM/State+Diagrams
     * The CorrelationId is intended to be used by CommandMessageListeners to
     * correlate messages.
     * The CorrelationId is used internally to route CommandReply messages to the
     * appropriate CommandOriginator
     * 
     * @return The Correlation Id for this message
     */
    public UUID getCorrelationId() {
        return correlationId;
    }
    @Deprecated
    public String getCorrelId(){
        return getCorrelationId().toString();
    }
    
}