package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.lsst.ccs.bus.states.AlertState;

/**
 * An Alert history for an Alert that has been raised.
 * This class is meant to be sent over the buses as part of an AlertSummary object.
 * It is built from an Alert.
 * 
 * A RaisedAlertHistory additionally contains list of all the RaisedAlertInstances of
 * when the Alert was raised.
 *
 * @author The LSST CCS Team.
 *
 */
public final class RaisedAlertHistory implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 883621057372902934L;

    private Alert latestAlert;
    private final List<RaisedAlertInstance> instances;
    private AlertState highestSeverity = AlertState.NOMINAL;

    /**
     * The main constructor of the RaisedAlert.
     *
     */
    public RaisedAlertHistory() {
        this.instances = new ArrayList<>();
    }
    
    /**
     * Copy constructor.
     * @param other Instance to be cloned.
     */
    public RaisedAlertHistory(RaisedAlertHistory other) {
        latestAlert = other.latestAlert;
        instances = new ArrayList<>(other.instances);
        highestSeverity = other.highestSeverity;
    }

    /**
     * Get the original Alert that was raised.
     * 
     * @return The original Alert that was raised.
     */
    public Alert getLatestAlert() {
       return latestAlert;
    }

    private void updateLatestAlert(Alert alert) {
        latestAlert = new Alert(alert.getAlertId(), alert.getDescription());        
    }
    
    /**
     * Add a new instance of this Alert.
     * 
     * @param severity The AlertState of the Alert.
     * @param alert The Alert that was raised.
     * @param timestamp The CCS timestamp of when the Alert was raised.
     */
    public void addAlertInstance(AlertState severity, Alert alert, long timestamp) {
        //REVIEW: This is currently assuming that the instances
        //are already added in the right order.
        //Should we sort this list based on the RaisedAlertInstance?
        if ( severity.compareTo(highestSeverity) > 0 ) {
            highestSeverity = severity;
        }
        instances.add(new RaisedAlertInstance(severity, timestamp));
        updateLatestAlert(alert); // REVIEW: not sure what the intention is here.
    }

    /**
     * Add a new instance of this Alert.
     * The current CCS timestamp is used.
     * 
     * @param severity The AlertState of the Alert.
     * @param alert The Alert that was raised.
     */
    public void addAlertInstance(AlertState severity, Alert alert) {
        addAlertInstance(severity,alert, System.currentTimeMillis());
    }
    
    /**
     * Get the number of times this Alert was raised.
     * 
     * @return the number of times the Alert was raised.
     */
    public int getNumberOfInstances() {
        return instances.size();
    }
    
    /**
     * Get the latest AlertState of this Alert.
     * 
     * @return Get the AlertState of the last time this Alert was raised.
     */
    public AlertState getLatestAlertState() {
        switch (getNumberOfInstances()) {
            case 0:
                return AlertState.NOMINAL;
            default:
                return instances.get(getNumberOfInstances()-1).getAlertState();                
        }
    }
    
    /**
     * Get the latest CCS timestamp of this Alert.
     * 
     * @return Get the CCS timestamp of the last time this Alert was raised.
     */
    public long getLatestAlertTimestamp() {
        switch (getNumberOfInstances()) {
            case 0:
                return -1;
            default:
                return instances.get(getNumberOfInstances()-1).getTimestamp();
        }
    }
    
    /**
     * Get the highest AlertState raised for this Alert.
     * 
     * @return The highest AlertState for this Alert.
     */
    public AlertState getHighestAlertState() {
        return highestSeverity;
    }
    
    /**
     * Get the full history of RaisedAlertInstances for this Alert.
     * 
     * @return The List containing the RaisedAlertInstances for this Alert.
     */
    public List<RaisedAlertInstance> getRaisedAlertInstancesList() {
        return Collections.unmodifiableList(instances);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Raised Alarms for id ").append(getLatestAlert().getAlertId()).append("\n");
        sb.append("Overall Severity: ").append(getHighestAlertState());
        return sb.toString();
    }
    
}
