package org.lsst.ccs.bus.utils;

import java.io.*;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.EncodedDataStatus;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * Utility class to code/decode objects.
 *
 * @author The LSST CCS Team
 * REVIEW: This encoding class is only used by StatusMessage. Should it be renamed?
 * Should it be part of StatusMessage?
 *
 */
public class EncodingUtils {

    private static final Logger logger = Logger.getLogger("org.lsst.ccs.bus");

    /**
     * An immutable class to describe the Class for a given object.
     * It basically collects the map of fields that can be decoded for a given object.
     * These objects are cached locally for a more efficient encoding.
     *
     */
    private static class ClassDescriptor {

        /**
         * The coding needs the list of fields and their name (when encoding key-value pairs).
         * the decoding needs to link a field to a name (hence the Map).
         * So a LinkedHashMap provides both access through list and map.
         */
        final LinkedHashMap<String, Field> mapFields;

        /**
         * simple constructor for immutable value object
         *
         * @param fieldList
         */
        ClassDescriptor(List<Field> fieldList) {
            if (fieldList != null) {
                this.mapFields = new LinkedHashMap<>();
                for (Field field : fieldList) {
                    mapFields.put(field.getName(), field);
                }
            } else {
                mapFields = null;
            }
        }

    }

    /**
     * local cache for <TT>ClassDescriptor</TT> objects
     */
    private final static ConcurrentHashMap<Class, ClassDescriptor> classMap = new ConcurrentHashMap<>();

    /**
     * Array of known types that don't need to be encoded.
     *
     */
    private static final Class[] WELL_KNOWN_TYPES = {
        Boolean.class,
        Boolean.TYPE,
        Integer.class,
        Integer.TYPE,
        Float.class,
        Float.TYPE,
        Double.class,
        Double.TYPE,
        Long.class,
        Long.TYPE,
        String.class,
        byte[].class,
        boolean[].class,
        int[].class,
        float[].class,
        double[].class,
        long[].class,
        String[].class,
        // dangerous : Serializable[].class,
        BitSet.class,
        BigDecimal.class,
        List.class,
        Map.class
    };

    /**
     * The NULL_DESCRIPTOR is used to describe objects that are not going to be encoded.
     *
     */
    private static final ClassDescriptor NULL_DESCRIPTOR = new ClassDescriptor(null) {
        @Override
        public String toString() {
            return "NULL_DESCRIPTOR";
        }
    };

    /**
     * The ENUM_DESCRIPTOR is specific to all enums.
     * Enums will be encoded in a special way: the key-value list will have an empty key and a String value
     * that represents the enums instance.
     */
    private static final ClassDescriptor ENUM_DESCRIPTOR = new ClassDescriptor(null) {
        @Override
        public String toString() {
            return "ENUM_DESCRIPTOR";
        }
    };

    /**
     * All the WELL_KNOWN_TYPES are assigned the NULL_DESCRIPTOR so that they
     * will not be encoded.
     *
     */
    static {
        for (Class clazz : WELL_KNOWN_TYPES) {
            classMap.put(clazz, NULL_DESCRIPTOR);
        }
    }

    /**
     * gets a <TT>ClassDescriptor</TT> object for a given class.
     * The descriptor cache is read and if no descriptor is found a new one is built
     * and inserted into the cache.
     *
     * TO-DO: Is it necessary to have this method synchronized?
     *
     * @param clazz The class of the Object for which we are getting the descriptor.
     * @return The ClassDescriptor for the provided Class.
     *
     */
    private synchronized static ClassDescriptor getDescriptorFor(Class clazz) {
        if (Map.class.isAssignableFrom(clazz)) {
            return NULL_DESCRIPTOR;
        }
        ClassDescriptor first = classMap.get(clazz);
        if (first == null) {
            first = buildDescriptorFor(clazz);
            classMap.put(clazz, first);
        }
        return first;
    }

    /**
     * Builds a <TT>ClassDescriptor</TT>a for a given class if it was not already built.
     *
     * @param clazz The Class for which to build the ClassDescriptor.
     * @return The ClassDescriptor for the provided class.
     *
     */
    private static synchronized ClassDescriptor buildDescriptorFor(Class clazz) {
        if (clazz.isEnum()) {
            return ENUM_DESCRIPTOR;
        }
        ArrayList<Field> listFields = new ArrayList<>();
        //Get the superclass and find all the inherited fields.
        Class superClazz = clazz.getSuperclass();
        if (superClazz != null) {
            if (!Object.class.equals(superClazz)) {
                ClassDescriptor superDescriptor = getDescriptorFor(superClazz);
                if (superDescriptor != NULL_DESCRIPTOR && superDescriptor.mapFields != null) {
                    listFields.addAll(superDescriptor.mapFields.values());
                }
            }
        }
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            int modifiers = field.getModifiers();
            if (Modifier.isStatic(modifiers)) {
                continue;
            }
            /*
             THis code has been added because some classes (such as ArrayList) use transient only for implementation reasons.
             so the transient field cannot be used for encoding/decoding ....  this is to circumvent this feature...
             */
            // if transient and of complex type: skip? examples : Thread, IO ... and so on (transient and not Serializable)
            if (Modifier.isTransient(modifiers)) {
                if (!Serializable.class.isAssignableFrom(field.getType())) {
                    continue;
                }
            }
            field.setAccessible(true);
            listFields.add(field);
        }
        ClassDescriptor res = new ClassDescriptor(listFields);
        return res;
    }

    /**
     * Encodes an object and return the DataList object containing all the KeyValueData objects
     * in which this Object has been decomposed.
     *
     * @param msg
     * @param value The Object to be decomposed.
     * @return The KeyValueDataList containing the KeyValueData in which this Object has been decomposed.
     */
    //TODO When EncodedDataStatus is removed we can get rid of the StatusMessage from 
    //the method signature. Also the Object will be a KeyValueList.
    public final static KeyValueDataList encodeObject(StatusMessage msg, Object value) {
        String key = "";
        long t = value instanceof KeyValueData ? ((KeyValueData)value).getTimestamp() : msg.getTimeStamp();
        Object obj = innerEncodeObject(value, key, t);
        if (obj instanceof KeyValueDataList) {
            return (KeyValueDataList) obj;
        }
        if (obj == null) {
            return null;
        }
        KeyValueDataList d = new KeyValueDataList();
        addObjectToDataList(key, obj, d, msg.getTimeStamp(), KeyValueData.KeyValueDataType.KeyValueTrendingData);
        return d;
    }

    protected final static Object innerEncodeObject(Object value, String key, long timestamp) {
        if (value == null) {
            return null;
        }
        Class clazz = value.getClass();
        ClassDescriptor descriptor = getDescriptorFor(clazz);

        if (value instanceof Collection) {
            Collection c = (Collection) value;
            KeyValueDataList dataList = new KeyValueDataList();
            for (Object val : c) {
                Object o = innerEncodeObject(val, key, timestamp);
                addObjectToDataList(key, o, dataList, timestamp, KeyValueData.KeyValueDataType.KeyValueTrendingData);
            }
            return dataList;
        }

        boolean isMap = value instanceof Map;
        if (isMap) {
            Map map = (Map) value;
            KeyValueDataList dataList = new KeyValueDataList();
            for (Object obj : map.keySet()) {
                String name = updateName(obj.toString(), key);
                Object o = innerEncodeObject(map.get(obj), name, timestamp);
                addObjectToDataList(name, o, dataList, timestamp, KeyValueData.KeyValueDataType.KeyValueTrendingData);
            }
            return dataList;
        }

        if (descriptor == NULL_DESCRIPTOR) {
            return value;
        }
        if (descriptor == ENUM_DESCRIPTOR) {
            return String.valueOf(value);
        }

        //TODO: Remove this block of code when EncodedDataStatus is removed.
        //It's just for backward compatibility
        if (value instanceof EncodedDataStatus) {
            KeyValueDataList res = new KeyValueDataList();
            EncodedDataStatus encodedStatus = (EncodedDataStatus) value;
            for (EncodedDataStatus status : encodedStatus) {
                addObjectToDataList(status.getKey(), innerEncodeObject(status.getEmbeddedObject(), status.getKey(), status.getDataTimestamp()), res, status.getDataTimestamp(), KeyValueData.KeyValueDataType.KeyValueTrendingData);
            }
            return res;
        }

        if ( value instanceof KeyValueData ) {
            KeyValueDataList res = new KeyValueDataList();
            KeyValueData kvd = ((KeyValueData)value);
            timestamp = kvd.getTimestamp();
            String name = updateName(kvd.getKey(), key);
            addObjectToDataList(name, innerEncodeObject(kvd.getValue(), name, timestamp), res, timestamp, kvd.getType());
            return res;
        }
        Collection<Field> descriptorValues = descriptor.mapFields.values();
        KeyValueDataList res = new KeyValueDataList();
        if (value instanceof Exception) {
            ((Exception) value).printStackTrace();
        }
        for (Field field : descriptorValues) {
            try {
                String name = updateName(field.getName(), key);
                Object val = innerEncodeObject(field.get(value), name, timestamp);
                addObjectToDataList(name, val, res, timestamp, KeyValueData.KeyValueDataType.KeyValueTrendingData);
            } catch (IllegalArgumentException | IllegalAccessException e) {
                throw new RuntimeException("Something went really wrong encoding the object.", e);
            }
        }
        return res;
    }

    private static String updateName(String name, String prefix) {
        String tmpName = name;
        if (prefix != null && !prefix.isEmpty()) {
            tmpName = prefix + "/" + tmpName;
        }
        return tmpName;
    }

    private static void addObjectToDataList(String name, Object val, KeyValueDataList res, long timestamp, KeyValueData.KeyValueDataType type) {
        if (val instanceof KeyValueDataList) {
            for ( KeyValueData d : (KeyValueDataList) val) {
                res.addData(d);
            }
        } else if (val instanceof KeyValueData) {
            res.addData((KeyValueData) val);
        } else {
            res.addData(name, (Serializable)val, timestamp, type);
        }
    }

}
