package org.lsst.ccs.bus.states;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.time.Instant;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.utilities.time.CCSInstantFactory;

/**
 * Contains a set of subsystem states, each represented by {@code Enum}. 
 * States are stored internally in a {@code ConcurrentHashMap}, mapped by their class names.
 *
 * This class is meant to be published on the buses with every status message
 * and when there is any state change.
 * 
 * This class is thread safe to the extent supported by the underlying {@code ConcurrentHashMap}:
 * no concurrent modifications are thrown, but atomicity of operations that involve
 * reading or modifying more than one elemental state is not guaranteed.
 *
 * @author The LSST CCS Team
 */
public class StateBundle implements Serializable, Cloneable {

    private Map<String, Enum> allStates = new ConcurrentHashMap<>();
    private Instant lastModified = CCSInstantFactory.now();

    /** Constructs a StateBundle from a set of states.
     * @param states The set of States to construct the StateBundle with.
     */
    public StateBundle(Enum... states) {
        for (Enum state : states) {
            innerSetState(state);
        }
        updateLastModified();
    }

    /** Copy constructor. */
    private StateBundle(StateBundle b) {
        this.allStates.putAll(b.allStates);
        this.lastModified = b.lastModified;
    }

    /**
     * Merge the content of two StateBundle objects returning an updated cloned
     * version of the original StateBundle object.
     * The original StateBundle is left unchanged.
     * 
     * @param newState The StateBundle with the changes.
     * @return An update cloned version of the original StateBundle.
     *
     */
    public StateBundle mergeState(StateBundle newState) {
        StateBundle clone = this.clone();
        clone.allStates.putAll(newState.allStates);
        clone.updateLastModified();
        return clone;
    }

    /**
     * Finds what has changed.
     * 
     * @param oldState The StateBundle of the previous state .
     * @return a new StateBundle containing only the changed states.
     */
    public StateBundle diffState(StateBundle oldState) {
        StateBundle diff = new StateBundle();
        for (Enum e : allStates.values()) {
            if (!oldState.isInState(e)) {
                diff.innerSetState(e);
            }
        }
        diff.updateLastModified();
        return diff;
    }

    /**
     * Sets the specified state to the bundle, replacing any previous state of the same type.
     * @param state The state to set on the StateBundle.
     */
    public final void setState(Enum state) {
        innerSetState(state);
        updateLastModified();
    }

    /**
     * Sets specified states to the bundle, replacing any previous states of the same types. 
     * @param states The specified states to set on the StateBundle
     */
    public final void setState(Enum... states) {
        for (Enum state : states) innerSetState(state);
        updateLastModified();
    }
    
    //Sets the state without updating the timestamp of the last update.
    private void innerSetState(Enum state) {
        allStates.put(state.getClass().getSimpleName(), state);
    }
    
    /**
     * Returns the current value for the specified State.
     *
     * @param <T> An {@code Enum} that defines a set of states.
     * @param enumClass The class of the {@code Enum} representing the state.
     * @return The current value for the given state.
     */
    public final <T extends Enum<T>> Enum getState(Class<T> enumClass) {
        return allStates.get(enumClass.getSimpleName());
    }

    /**
     * Check if this {@code StateBundle} contains the specified state.
     *
     * @param <T> An {@code Enum} that defines a set of states.
     * @param state The value of the {@code Enum} to check.
     * @return true if this {@code StateBundle} contains the provided state
     */
    public <T extends Enum<T>> boolean isInState(T state) {
        Enum e = getState(state.getClass());
        return e == null ? false : e.equals(state);
    }

    /**
     * Check if this StateBundle is in all the states of a given StateBundle.
     * 
     * @param stateBundle The StateBundle containing all the states to check.
     * @return true if this StateBundle is in all the states of the provided one.
     */
    public boolean isInState(StateBundle stateBundle) {
        for (Enum e : stateBundle.allStates.values()) {
            if (!isInState(e)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Get the {@code Instant} of the last time the StateBundle
     * was modified.
     * @return The {@code Instant} of the last change
     */
    public Instant getLastModified() {
        return lastModified;
    }
    
    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof StateBundle)) {
            return false;
        }

        StateBundle state = (StateBundle) o;
        if ( allStates.size() != state.allStates.size() ) {
            return false;
        }

        for (String c : allStates.keySet()) {
            if (!state.isInState(allStates.get(c))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode() {
        int result = 0;
        for (Enum e : allStates.values()) {
            result = 31 * result + e.hashCode();
        }
        return result;
    }

    @Override
    public StateBundle clone() {
        StateBundle b = new StateBundle(this);
        return b;
    }

    @Override
    public String toString() {

        String s = super.toString() + " ";
        for (Entry<String, Enum> entry : allStates.entrySet()) {
            s += entry.getKey() + ":" + entry.getValue() + ",";
        }
        return s.substring(0, s.length() - 1);
    }

    private void updateLastModified() {        
        lastModified = CCSInstantFactory.now();        
    }
    
    // to silently ignore enums for which the class is locally unknown

    private void writeObject(ObjectOutputStream out) throws IOException {
        out.writeInt(allStates.size());
        for (Enum e : allStates.values()) {
            out.writeObject(e.getClass().getName());
            out.writeObject(e.toString());
        }
        out.writeObject(getLastModified());
    }

    private void readObject(ObjectInputStream in) throws IOException,
            ClassNotFoundException {
        int n = in.readInt();
        allStates = new ConcurrentHashMap<String, Enum>(n);
        for (int i = 0; i < n; i++) {
            String className = (String) in.readObject();
            String value = (String) in.readObject();

            try {
                Class<Enum> c = (Class<Enum>) Class.forName(className);
                Method m = c.getDeclaredMethod("valueOf", String.class);
                Enum e = (Enum) m.invoke(null, value);
                allStates.put(c.getSimpleName(), e);
            } catch (Exception e) {
                // silently ignore
            }

        }
        lastModified = (Instant)in.readObject();
    }
}
