package org.lsst.ccs.bus.messages;

import java.io.Serializable;
import org.lsst.ccs.bus.states.StateBundle;

/**
 * Base class for all messages sent on the Status Bus.
 *
 * All StatusMessage objects contain the internal state of the Agent from which
 * they originated at the time of publication.
 *
 * The default encoding for StatusMessages breaks down the embedded object in
 * key-value pairs (and possibly timestamp) and returns the DataList object containing them.
 * Subclasses can change this default encoding behavior. Please refer to their
 * documentation for further details on their encoding strategy.
 *  
 * @author LSST CCS Team
 * @param <T> Any Serializable class embedded in the StatusMessage
 * @param <D> The class of the encoded object.
 * 
 */
public abstract class StatusMessage<T extends Serializable,D> extends BusMessage<T,D> {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 6763213329636909L;

    //TODO Make this variable final when setState method is removed.
    private StateBundle state;

    /**
     * Build a StatusMessage from the provided Object.
     *
     * @param obj The Serializable object to be sent over the buses.
     * @param state The State of the Agent at the time the object is created.
     */
    public StatusMessage(T obj, StateBundle state) {
        super(obj);
        /**
         * Implementation note: the state object corresponds to the state of the
         * Agent when the StatusMessage was created. To make sure it
         * does not change externally, we keep a clone of the original StateBundle.
         *
         */        
        this.state = state != null ? state.clone() : null;
    }

    /**
     * Set the StateBundle object containing the Agent's internal state at
     * the moment the StatusMessage was published.
     * This method should be only used by the framework.
     * REVIEW: can the state be provided at creation?
     *
     * @param state The StateBundle object.
     *
     */
    @Deprecated
    public void setState(StateBundle state) {
        /**
         * Implementation note: the state object corresponds to the state of the
         * Agent when the StatusMessage was created. To make sure it
         * does not change externally, we keep a clone of the original StateBundle.
         *
         */
        this.state = state.clone();
    }

    //TODO REMOVE THIS METHOD
    //Adding this method so that it is possible to get the byte[] array at the
    //StatusMessage level. This is needed to keep backward compatibility with the
    //SerializedDataStatusListener.
    //The method should be removed when SerializedDataStatusListener is removed.
    @Deprecated
    public byte[] getSerializedObject() {
        return super.getSerializedObject();
    }
      
    /**
     * Get the published State object that contains the Agent's internal state
     * at the moment the StatusMessage was published.
     *
     * @return The StateBundle object.
     */
    public StateBundle getState() {
        return state;
    }

    @Override
    public String toString() {
        String toString = super.toString();
        toString += " state=" + state + "\n";
        return toString;
    }

}
