package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.states.ConfigurationState;

/**
 * A Class containing the Agent's configuration information.
 * 
 * @author The LSST CCS Team
 */
//TODO: Make this class final when corresponding class in org.lsst.ccs.bus.messages is removed
public class ConfigurationInfo implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 349664303742923784L;
    
    protected final String descriptionName;
    protected final Map<String, Boolean> hasCategoryChanged = new HashMap<>();
    protected final Map<String,String> tags = new HashMap<>();
    private final ConfigurationState configState;
    
    private final List<ConfigurationParameterInfo> parametersView = new ArrayList<>();
    
    protected ConfigurationInfo(ConfigurationState configState, String tagName){
        this.configState = configState;
        this.descriptionName = tagName;
    }
    
    public ConfigurationInfo(ConfigurationState configState, String tagName, Map<String, String> tags, Map<String, Boolean> hasCategoryChanged, List<ConfigurationParameterInfo> parametersView){
        this(configState, tagName);
        this.tags.putAll(tags);
        this.hasCategoryChanged.putAll(hasCategoryChanged);
        this.parametersView.addAll(parametersView);
    }

    /**
     * Returns the subsystem description name, ie the name of the description
     * file the subsystem is started with.
     * @return the subsystem description name
     */
    public String getDescriptionName() {
        return descriptionName;
    }

    /**
     * @return the String representation of the configuration state 
     */
    public String getConfigurationName(){
        String res = descriptionName+"[";
        res+=getTaggedCategories()+"]";
        return res;
    }
    
    
    /**
     * Returns the String representation of the categories
     * ex : "catA:A1,catB:B2*,def4"
     * @return 
     */
    private String getTaggedCategories(){
        
        String res = "";
        for (Map.Entry<String,String> entry : new TreeMap<String,String>(tags).entrySet()){
            boolean hasChanges = hasChangesForCategory(entry.getKey());
            res += (entry.getKey().isEmpty()?"":(entry.getKey()+":"))
                    + entry.getValue()+(hasChanges?"*":"")+",";
        }
        if (!res.isEmpty()){
            res = res.substring(0, res.length()-1);
        }
        return res;
    }

    /**
     * Returns true if there are unsaved changes for the specified category.
     * @param category
     * @return true if there are unsaved changes for the specified category,
     * false otherwise.
     */
    public boolean hasChangesForCategory(String category){
        return hasCategoryChanged.get(category);
    }
    
    /**
     * 
     * @return true if there are no unsaved changes for all categories.
     */
    public boolean hasChanges(){
        for(boolean b : hasCategoryChanged.values()){
            if (b) return true;
        }
        return false;
    }
    
    /**
     * @param category
     * @return the base configuration name for the specified category.
     */
    public String getConfigNameForCategory(String category) {
        return tags.get(category);
    }
    
    /**
     * Tests if {@category} is one of the subsystem parameter categories.
     * @param category
     * @return true if {@code category} is one of the parameter categories of 
     * the subsystem.
     */
    public boolean hasCategory(String category) {
        return tags.containsKey(category);
    }
    
    /**
     * @return the set of categories the configurable parameters of the subsystem
     * this {@configurationInfo} object stands for are splitted into.
     */
    public Set<String> getCategorySet() {
        return tags.keySet();
    }
    
    /**
     * @return the configuration state
     */
    public ConfigurationState getConfigurationState(){
        return configState;
    }
    
    /**
     * 
     * @return a String representation of the {@ConfigurationInfo} object
     */
    @Override
    public String toString() {
        return "Configuration state : "+getConfigurationName()
                + " Description " + getDescriptionName();
    }
    
    /**
     * Returns a list of {@code ConfigurationParameterInfo}. Each element of this
     * list represents the current view of a configurable parameter.
     * @return a List of {@code ConfigurationParameterInfo}
     */
    public List<ConfigurationParameterInfo> getAllParameterInfo(){
        return Collections.unmodifiableList(parametersView);
    }
    
    // Common actions on the list of parameter info

    /**
     * Returns the current value of each parameter that belong to the specified
     * category.
     * @param category
     * @return a Map of parameter pathName and their current value.
     */
    public Map<String,String> getCurrentValuesForCategory(String category){
        return Collections.unmodifiableMap(
                parametersView.stream()
                        .filter(parmInfo -> parmInfo.getCategoryName().equals(category))
                        .collect(Collectors.toMap(ConfigurationParameterInfo::getPathName, ConfigurationParameterInfo::getCurrentValue))
        );
    }
    
    /**
     * Returns the current value of each parameter that belong to the specified 
     * configurable component.
     * @param componentName the name of the {@Configurable}
     * @return a Map of parameter pathName and their current value. 
     */
    public Map<String,String> getCurrentValuesForComponent(String componentName){
        return Collections.unmodifiableMap(
                parametersView.stream()
                        .filter(parmInfo -> parmInfo.getPathName().startsWith(componentName+"/"))
                        .collect(Collectors.toMap(ConfigurationParameterInfo::getPathName, ConfigurationParameterInfo::getCurrentValue))
        );
    }
    
    /**
     * Return the current value of a given configurable parameter
     * @param pathName the path name of a parameter
     * @return a String representation of the parameter's value
     */
    public String getCurrentValueForParameter(String pathName){
        return parametersView.stream()
                .filter(parmInfo -> parmInfo.getPathName().equals(pathName))
                .map(ConfigurationParameterInfo::getCurrentValue).findFirst().get();
    }
}
