package org.lsst.ccs.bus.states;

import java.io.Serializable;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map.Entry;

/**
 * A class to collect any number of State information.
 * It is designed to accept any state in the form of an Enum. It will be stored
 * internally in a HashMap using the class of the Enum as key. So at any given
 * point this class will store the current value of all the provided states.
 *
 * This class is meant to be published on the buses with every status message and
 * when there is any state change.
 *
 * @author The LSST CCS Team
 */
public class StateBundle implements Serializable, Cloneable {

    private final HashMap<String, Enum> allStates = new LinkedHashMap<>();

    /**
     * Construct a StateBundle from a set of states.
     * @param states A set of states.
     * 
     */
    public StateBundle(Enum... states) {
        for (Enum e : states) {
            setState(e);
        }
    }

    private StateBundle(StateBundle b) {
        this.allStates.putAll(b.allStates);
    }

    /**
     * Merge the content of two StateBundle objects returning an updated cloned
     * version of the original StateBundle object.
     *
     * @param newState The StateBundle with the changes.
     * @return An update cloned version of the original StateBundle.
     *
     */
    public StateBundle mergeState(StateBundle newState) {
        StateBundle clone = this.clone();
        for (Enum e : newState.allStates.values()) {
            clone.setState(e);
        }
        return clone;
    }

    /**
     * Set the value of a state in the form of an Enum value.
     * The value will be stored in a HashMap with the Enum's class as key.
     *
     * @param <T> An enum that defines a set of states.
     * @param state The current value for the state.
     *
     */
    public final <T extends Enum<T>> void setState(T state) {
        allStates.put(state.getClass().getSimpleName(), state);
    }

    /**
     * Get the current value for a given State by providing the class of the
     * Enum representing the state.
     *
     * @param <T> An enum that defines a set of states.
     * @param enumClass The class of the Enum representing the state.
     * @return The current value for the given state.
     *
     */
    public final <T extends Enum<T>> Enum getState(Class<T> enumClass) {
        return allStates.get(enumClass.getSimpleName());
    }

    /**
     * Check if this StateBundle contains a given Enum state.
     *
     * @param <T> An enum that defines a set of states.
     * @param state The value of the Enum to check.
     * @return true if this StateBundle contains the provided state
     *
     */
    public <T extends Enum<T>> boolean isInState(T state) {
        Enum e = getState(state.getClass());
        return e == null ? false : e.equals(state);
    }

    /**
     * Check if this StateBundle is in all the states of a given StateBundle.
     * @param stateBundle The StateBundle containing all the states to check.
     * @return true if this StateBundle is in all the states of the provided one.
     */
    public boolean isInState(StateBundle stateBundle) {
        for ( Enum e : stateBundle.allStates.values() ) {
            if ( ! isInState(e) ) {
                return false;
            }
        }
        return true;
    }

    /** MT: Commented out. What is the return value if the StateBundle
     * Does not have the provided state? getState would return null.
     * How would it compare?
     */
//    public <T extends Enum<T>> int compareState(T state) {
//        return getState(state.getClass()).compareTo(state);
//    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof StateBundle)) {
            return false;
        }

        StateBundle state = (StateBundle) o;

        for (String c : allStates.keySet()) {
            if (!state.isInState(allStates.get(c))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode() {
        int result = 0;
        for (Enum e : allStates.values()) {
            result = 31 * result + e.hashCode();
        }
        return result;
    }

    @Override
    public StateBundle clone() {
        StateBundle b = new StateBundle(this);
        return b;
    }

    @Override
    public String toString() {

        String s = super.toString() + " ";
        for (Entry<String, Enum> entry : allStates.entrySet()) {
            s += entry.getKey() + ":" + entry.getValue() + ",";
        }
        return s.substring(0, s.length() - 1);
    }

    
}
