
package org.lsst.ccs.bus.messages;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A Class containing the Agent's configuration information.
 * 
 * @author The LSST CCS Team
 */
public class ConfigurationInfo implements Serializable {

    
    private static final Pattern pattern = Pattern.compile("([a-zA-Z0-9-_]++)(\\[(.+)\\])?");
    
    private final String descriptionName, tagName, fullConfigurationName;
    private final boolean hasLocalChanges;
    private final Map<String,String> tags = new HashMap<>();
    
    public ConfigurationInfo(String fullConfigurationName) {
        this.fullConfigurationName = fullConfigurationName;
        this.hasLocalChanges = fullConfigurationName.contains("*");
        Matcher m = pattern.matcher(fullConfigurationName);
        if ( ! m.matches() ) {
            throw new IllegalArgumentException("Full Configuration Name "+fullConfigurationName+" does not match the conventional standard.");
        }
        this.descriptionName = m.group(1);
        this.tagName = m.group(3) != null ? m.group(3) : "";
        
        if ( ! tagName.isEmpty() ) {
            StringTokenizer st = new StringTokenizer(tagName,",");
            while ( st.hasMoreTokens() ) {
                String tag = st.nextToken();
                if ( ! tag.contains(":") ) {
                    tag = ":"+tag;
                }
                int index = tag.indexOf(":");
                tags.put(tag.substring(0,index), tag.substring(index+1));
            }
        }
        
    }
    
    public String getDescriptionName() {
        return descriptionName;
    }

    public String getTagName() {
        return tagName;
    }

    public String getFullConfigurationName() {
        return fullConfigurationName;
    }

    public boolean hasLocalChanges() {
        return hasLocalChanges;
    }
    
    public String getTagForCategory(String category) {
        return tags.get(category);
    }
    
    public boolean hasCategory(String category) {
        return tags.containsKey(category);
    }
    
    public Set<String> getCategorySet() {
        return tags.keySet();
    }
    
    @Override
    public String toString() {
        return "FullConfigurationName: "+getFullConfigurationName()+" Description "+getDescriptionName()+" Tag "+getTagName();
    }
                
}
