package org.lsst.ccs.bus.messages;

import java.io.Serializable;
import java.util.Objects;

/**
 * Base class for messages sent on the buses.
 * Concrete implementations (command, status and log messages) will be sent
 * on the appropriate bus.
 * 
 * NOTE: Can we remove some or all the setter methods?
 * NOTE: This is currently an interface and it will replace the implementation of
 * org.lsst.ccs.bus.BusMessage
 *
 * @author LSST CCS Team
 */
public abstract class BusMessage implements Serializable {

    private String summary;
    private AgentInfo originAgentInfo;
    private final long timeStamp = System.currentTimeMillis();
    
    /**
     * The Origin of the BusMessage, the bus registration name of the Agent from which it
     * originated.
     *
     * @return The origin of this BusMessage.
     */
    public AgentInfo getOriginAgentInfo(){
        return originAgentInfo;
    }

    /**
     * Sets the origin for this BusMessage.
     * This method should only be used internally, by the MessagingLayer Architecture.
     * It must not be used by users/developers.
     *
     * @param agent The origin of the BusMessage.
     *
     */
    public final void setOriginAgentInfo(AgentInfo agent){
        if ( originAgentInfo != null ) {
            throw new RuntimeException("The setOriginAgentInfo method on BusMessage must be invoked only once!!!");
        }
            
        this.originAgentInfo = agent;
    }

    /**
     * Get the timestamp associated to this BusMessage. This is in milliseconds since 1970.
     * Is this the time when the message was sent or created? Right now it's created.
     *
     * @return The BusMessage timestamp.
     *
     */
    public long getTimeStamp(){
        return timeStamp;
    }

    /**
     * The summary string for this BusMessage.
     * This is currently used only for Alarms.
     * Could this be used by BusMessage listeners to display
     * summary information about received BusMessages?
     *
     * @return The summary string for this BusMessage
     */
    public String getSummary(){
        return summary;
    }

    /**
     * Set the summary for this BusMessage.
     * Who should use this method?
     * 
     * @param summary The summary for this BusMessage.
     */
    public void setSummary(String summary){
        this.summary = summary;
    }

    @Override
    public int hashCode() {
        int hash = 5;
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final BusMessage other = (BusMessage) obj;
        if (!Objects.equals(this.summary, other.summary)) {
            return false;
        }
        if (!Objects.equals(this.originAgentInfo, other.originAgentInfo)) {
            return false;
        }
        if (this.timeStamp != other.timeStamp) {
            return false;
        }
        return true;
    }



}
