package org.lsst.ccs.bus.messages;

import org.lsst.ccs.utilities.logging.Logger;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.lsst.ccs.bus.messages.StatusCodec.CodedData;

/**
 * A  key-value pair.
 * The value can be a specific <TT>CodedData</TT>
 * that codes data that is not of a "well known type".
 * @author bamade
 */
// Date: 23/12/2013

public class KeyData implements Serializable {

    public static Logger logger = Logger.getLogger("org.lsst.ccs.bus") ;

    private final String key;
    private final Object data;

    KeyData(String key, Object data) {
        this.key = key;
        this.data = data;
    }

    public String getKey() {
        return this.key;
    }


    /**
     * returns a list of key-values that can be directly used byt a trending database:
     * each key (the name of fields in a composite object) is prepended with the actual key
     * of this object (if overall key is "position" you may get something like "position.x" and "position.y"
     * as keys in the list)
     *
     * @return
     */
    public List<KeyData> getContentAsList() {
        return StatusCodec.asSimpleKeyValueList(key, data);
    }

    /**
     * returns the value of the contained object if its class is accessible to the current
     * ClassLoader. Otherwise the <TT>Optional</TT> object is empty.
     * @return
     */
    public Optional<Object> getValue() {
        Object obj = null;
        try {
            obj = StatusCodec.decode(data);
        } catch (Exception e) {
            //todo: log!
            logger.warn("can't decode " + getKey(),e) ;
        }
        if(obj == null) return Optional.empty() ;
        return Optional.of(obj);
    }


    /**
     * Big objects can be kept in database as Blobs.
     * Objects that are <TT>Serializable</TT> but not decomposable (in key-value list)
     * are stored in a byte array.
     * @implSpec to be decided: do we need to return a "crystallized" version of a decomposable object too?
     * @return
     */
    public Optional<byte[]> getCrystallizedData() {
        byte[] res = null;
        if (data instanceof CodedData) {
            CodedData obj = ((CodedData) data);
            if (obj.content instanceof byte[]) {
                res = (byte[]) obj.content;
            }
        }
        return Optional.ofNullable(res);
    }

    public String getUpdatedKey(String globalName) {
        if (this.key.length() == 0) {
            return globalName;
        } else {
            return globalName + '/' + this.key;
        }
    }

    public Object getRawValue() {
        return this.data;
    }

    @Override
    public String toString() {
        return "{" +
                "key='" + key + '\'' +
                ", data=" + data +
                '}';
    }
}
