package org.lsst.ccs.bus.messages;


import java.beans.ConstructorProperties;
import java.io.*;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.lang.reflect.Parameter;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;

import static org.lsst.ccs.bus.messages.KeyData.CodedData;

/**
 * utilities to code/decode DataStatus_Deprecated.
 * <p/>
 * TODO: this code is in a circular dependency with the KeyData class.
 * <p/>
 * TODO! make this class package friendly
 * It has to be public because there is a test in apptests that needs this.
 * We need to move this test to java in this package.
 * @author bamade
 */
// Date: 20/12/2013


public class StatusCodec {
    static Logger logger = Logger.getLogger("org.lsst.ccs.bus") ;
    /**
     * Immutable objects of this class describe what we can do with instances of a class.
     * They are kept in a local cache to fasten object encoding/decoding.
     */
    static class ClassDescriptor {
        /**
         * can we decompose objects of this class in a list of key-values ?
         */
        final boolean decomposable;
        /**
         * do we need to serialize the object in a byte array (like a Marshalled object)?
         * This can happen :
         * <UL>
         * <LI/> if and only if the object is <TT>Serializable</TT>
         * <LI/> if the object is not <TT>decomposable</TT>
         * <LI/> if the object is <TT>decomposable</TT> but has no usable constructor.
         * </UL>
         */
        final boolean toCrystallize;
        /**
         * if this object has a usable constructor here it is.
         * Note that it can be private or protected.
         * Can be null if the Class is not locally reachable by the ClassLoader
         * (when decoding)
         */
        final Constructor usableCtor;
        /**
         * if true use the no-arg ctor, if false use <TT>ConstructorProperties</TT>
         * annotations or new java 8 parameter names introspection.
         */
        final boolean isNoargCtor;
        /**
         * The super-class.
         * may be null when decoding (class not locally reachable by ClassLoader)
         */
        final Class superClass;
        //final List<Field> fieldList;// deprecated and mapFields replaces by a LinkedHashMap
        /**
         * The coding needs the list of fields and their name (when encoding key-value pairs).
         * the decoding needs to link a field to a name (hence the Map).
         * So a LinkedHashMap provides both access through list and map.
         */
        final LinkedHashMap<String, Field> mapFields;

        /**
         * simple constructor for immutable value object
         *
         * @param decomposable
         * @param toCrystallize
         * @param usableCtor
         * @param superClass
         * @param fieldList
         */
        ClassDescriptor(boolean decomposable, boolean toCrystallize, Constructor usableCtor, boolean useNoargCtor, Class superClass, List<Field> fieldList) {
            this.decomposable = decomposable;
            this.toCrystallize = toCrystallize;
            this.usableCtor = usableCtor;
            this.isNoargCtor = useNoargCtor;
            this.superClass = superClass;
            //this.fieldList = fieldList;
            if (fieldList != null) {
                this.mapFields = new LinkedHashMap<>();
                for (Field field : fieldList) {
                    mapFields.put(field.getName(), field);
                }
            } else {
                mapFields = null;
            }
        }

        @Override
        public String toString() {
            return "ClassDescriptor{" +
                    "decomposable=" + decomposable +
                    ", toCrystallize=" + toCrystallize +
                    ", usableCtor=" + usableCtor +
                    ", superClass=" + superClass +
                    ", fieldList=" + mapFields.values() +
                    '}';
        }
    }

    /**
     * local cache for <TT>ClassDescriptor</TT> objects
     */
    static ConcurrentHashMap<Class, ClassDescriptor> classMap = new ConcurrentHashMap<>();
    /**
     * predefined list of types where values won't be encoded either by "crystalizing"
     * or with a list of key-value.
     */
    public static final Class[] WELL_KNOWN_TYPES = {
            Boolean.class,
            Boolean.TYPE,
            Integer.class,
            Integer.TYPE,
            Float.class,
            Float.TYPE,
            Double.class,
            Double.TYPE,
            String.class,
            byte[].class,
            boolean[].class,
            int[].class,
            float[].class,
            double[].class,
            String[].class,
            // dangerous : Serializable[].class,
            BitSet.class,
            BigDecimal.class,
    };
    /**
     * This descriptor is used to describe data that will not be encoded (either with a key-value list or
     * with a crystallized byte array).
     */
    public static final ClassDescriptor NULL_DESCRIPTOR = new ClassDescriptor(false, false, null, false, null, null) {
        public String toString() {
            return "NULL_DESCRIPTOR";
        }
    };

    /**
     * This decriptor is common to all enums.
     * Enums will be encoded in a special way: the key-value list will have an empty key and a String value
     * that represents the enums instance, the object will also be crystallized (this is necessary if the client side
     * does not have the Enum class at hand).
     */
    public static final ClassDescriptor ENUM_DESCRIPTOR = new ClassDescriptor(true, true, null, false, null, null) {
        public String toString() {
            return "ENUM_DESCRIPTOR";
        }
    };

    /**
     * this initialisation block will mark all WELL_KNOWN_TYPES to be linked to a NULL_DESCRIPTOR
     */
    static {
        for (Class clazz : WELL_KNOWN_TYPES) {
            classMap.put(clazz, NULL_DESCRIPTOR);
        }
    }

    /**
     * gets a <TT>ClassDescriptor</TT> object for a given class.
     * The descriptor cache is read and if no descriptor found a new one is built
     * and inserted into the cache.
     * <BR/>
     * TODO: is it necessary to synchronize?
     *
     * @param clazz
     * @return
     */
    static ClassDescriptor getDescriptorFor(Class clazz) {
        ClassDescriptor first = classMap.get(clazz);
        if (first == null) {
            first = buildDescriptorFor(clazz);
            classMap.put(clazz, first);
        }
        return first;
    }

    /**
     * Builds a <TT>ClassDescriptor</TT>a for a given class
     *
     * @param clazz
     * @return
     * @implSpec <UL>
     * <LI>if it is an enum class the common <TT>ENUM_DESCRIPTOR</TT> is returned</LI>
     * <LI> does the class have a usable constructor? if yes it is kept and made accessible</LI>
     * <LI> we get the super class and if it is not Object :</LI>
     * <UL>
     * <LI> we get the descriptor for the superClass</LI>
     * <LI> if the descriptor for the superclass notes that it is to crystallize then the current descriptor
     * is not decomposable and is to crystallize. This may be changed in further versions when we decide that
     * crystallization is not incompatible with list decomposition</LI>
     * <LI> the fields of the superclass are added to the fields descriptions of the current class</LI>
     * </UL>
     * <LI> we add the fields to the list of fields in the descriptor.
     * if a field which is not an enum does not have a usable constructor then the current object is to be
     * crystallized. If a field (which is not an enum) is to be crystallized then the current object is
     * to be crystallized  </LI>
     * </UL>
     */
    private static synchronized ClassDescriptor buildDescriptorFor(Class clazz) {
        // if enum
        if (clazz.isEnum()) {
            return ENUM_DESCRIPTOR;
        }
        // is it known to everybody ?
        //todo: if startsWith "java" && Serializable -> NULL-DESCRIPTOR ?
        // todo: but beware of profiles in java8

        Constructor usableCtor = null;
        boolean isNoArgCtor = false;
        try { // first we go after a no-arg ctor
            usableCtor = clazz.getDeclaredConstructor();
            usableCtor.setAccessible(true);
            isNoArgCtor = true;
        } catch (NoSuchMethodException e) {
            //TODO: in fact not correct
            // we are supposing that ALL constructors are correct
            //then we go after ConstructorProperties
            Constructor[] ctors = clazz.getConstructors();
            if(ctors.length > 1) {
                logger.warning("encoding does not work properly when there is constructor overloading in class "+ clazz.getName()); ;
            } else {
                for (Constructor ctor : ctors) {
                    Annotation annotation = ctor.getAnnotation(ConstructorProperties.class);
                    if (annotation != null) {
                        usableCtor = ctor;
                        usableCtor.setAccessible(true);
                        isNoArgCtor = false;
                        break;
                    } else {
                        //TODO: bug! this only works if there is only one parameter
                        Parameter[] parms = ctor.getParameters();
                        if (parms[0].isNamePresent()) {
                            usableCtor = ctor;
                            usableCtor.setAccessible(true);
                            isNoArgCtor = false;
                            break;
                        }

                    }
                }
            }
        }
        boolean crystallize = (usableCtor == null);
        // introspect the class
        ArrayList<Field> listFields = new ArrayList<>();
        // get super class
        // if supserClass is not Object
        // call recursively getDescriptorFor
        // and complete the FieldList
        Class superClazz = clazz.getSuperclass();
        if(superClazz != null) { // the type is itself Object if superclass is null!!!
            if (!Object.class.equals(superClazz)) {
                ClassDescriptor superDescriptor =
                        getDescriptorFor(superClazz);
                if (superDescriptor.toCrystallize) {
                    //TODO : change that in future versions ?
                    return new ClassDescriptor(false, true, usableCtor, isNoArgCtor, superClazz, null);
                } else {
                    //listFields.addAll(superDescriptor.fieldList);
                    listFields.addAll(superDescriptor.mapFields.values());
                }
            }
        }
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            int modifiers = field.getModifiers();
            if (Modifier.isStatic(modifiers) ) {
                //if (Modifier.isStatic(modifiers) || Modifier.isTransient(modifiers)) {
                continue;
            }
            /*
            THis code has been added because some classes (such as ArrayList) use transient only for implementation reasons.
            so the transient field cannot be used for encoding/decoding ....  this is to circumvent this feature...
            */
            // if transient and of complex type: skip? examples : Thread, IO ... and so on (transient and not Serializable)
            if(Modifier.isTransient(modifiers)) {
             if(!   Serializable.class.isAssignableFrom( field.getType()))  {
                 continue;
             }
            }
            Class fieldType = field.getType();
            ClassDescriptor descriptor = getDescriptorFor(fieldType);
            // TO BE CHANGED IN THE FUTURE
            if (descriptor != NULL_DESCRIPTOR && descriptor != ENUM_DESCRIPTOR) {
                if (descriptor.toCrystallize) {
                    //TODO: not necessarily true: we may crystallize elements of a list?
                    crystallize = true;
                    break;// todo: same remark as above
                }
                if (descriptor.usableCtor == null) {
                    crystallize = true;
                    //usableCtor = null ;
                }
                /*
                */
            }
            field.setAccessible(true);
            listFields.add(field);
        }
        ClassDescriptor res = new ClassDescriptor(true, crystallize, usableCtor, isNoArgCtor, superClazz, listFields);
        return res;
    }

    /**
     * creates a list of key-value pairs (<TT>KeyData</TT>) to describe the content of an Object.
     * each value is encoded (but should not throw any exception at this stage of the implementation)
     *
     * @param object
     * @param descriptor
     * @return
     */
    static List<KeyData> listEncode(Object object, ClassDescriptor descriptor) {
        Collection<Field> descriptorValues = descriptor.mapFields.values();
        //List<KeyData> res = new ArrayList<>(descriptor.fieldList.size());
        List<KeyData> res = new ArrayList<>(descriptorValues.size());
        for (Field field : descriptorValues) {
            //for (Field field : descriptor.fieldList) {
            try {
                Object val = encode(field.get(object));
                res.add(new KeyData(field.getName(), val));
            } catch (Exception e) {
                //todo: to be changed if we authorize decomposable+toCrystallize
                //TODO LOG!
                System.err.println("OOPS " + e);
            }
        }
        return res;
    }

    /**
     * Encodes an object.
     * May build a list of key-value pairs and/or a crystallized value of this object if it is not
     * of a "well known type".
     *
     * @param value
     * @return the argument if it is of a "well known type" or a <TT>CodedData</TT> object
     * @throws IOException
     */
    public static Object encode(Object value) throws IOException {
        if(value == null) return null ;
        Class clazz = value.getClass();
        ClassDescriptor descriptor = getDescriptorFor(clazz);
        if (descriptor == NULL_DESCRIPTOR) {
            return value;
        }
        if (descriptor == ENUM_DESCRIPTOR) {
            CodedData codedEnum = new CodedData(clazz.getName());
            codedEnum.content = crystallize(value);
            List<KeyData> res = new ArrayList<>(1);
            res.add(new KeyData("", String.valueOf(value)));
            codedEnum.listKV = res;
            return codedEnum;
        }
        CodedData codedRes = new CodedData(clazz.getName());
        //if ((descriptor.usableCtor == null) && (Serializable.class.isAssignableFrom(clazz))) {
        //TODO: if ! decomposable ?
        // TODO : this class can be read but not deserialized
        //}
        if (descriptor.decomposable) {
            // decompose
            codedRes.listKV = listEncode(value, descriptor);
        }
        if (descriptor.toCrystallize) {
            codedRes.content = crystallize(value);

        }
        return codedRes;
    }

    /**
     * a creates a byte array containing a serialized object.
     *
     * @param value
     * @return
     * @throws IOException if the argument is not <TT>Serializable</TT>
     */
    static byte[] crystallize(Object value) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(bos);
        oos.writeObject(value);
        return bos.toByteArray();
    }

    /**
     * Decodes an object.
     * If the value is of a "well-known type" it is returned if it is <TT>CodedData</TT>
     * it is either deserialized from a crystallized state or rebuilt from the list of key-value pairs
     *
     * @param value
     * @return
     * @throws Exception
     */
    public static Object decode(Object value) throws Exception {
        if(value == null) return null ;
        if (value instanceof CodedData) {
            CodedData codedData = (CodedData) value;
            String className = codedData.className;
            Class clazz = Class.forName(className);
            ClassDescriptor descriptor = getDescriptorFor(clazz);
            // now can we build from ctor or deserialize?
            if (descriptor.toCrystallize) {
                byte[] array = (byte[]) codedData.content;
                ByteArrayInputStream bis = new ByteArrayInputStream(array);
                ObjectInputStream ois = new ObjectInputStream(bis);
                return ois.readObject();
            } else if (descriptor.usableCtor != null) {
                Object res = null;
                if (descriptor.isNoargCtor) {
                    res = descriptor.usableCtor.newInstance();
                    // which is best hashcode for descriptor or just read in sequence?
                    for (KeyData keyValue : codedData.listKV) {
                        String key = keyValue.getKey();
                        Field field = descriptor.mapFields.get(key);
                        if (field != null) {
                            Object valForField = decode(keyValue.getRawValue());
                            field.set(res, valForField);
                        }
                    }
                } else {
                    //TODO get the information once in the descriptor instead of doing it always again
                    String[] names ;
                    ConstructorProperties ctorProps = (ConstructorProperties) descriptor.usableCtor.getAnnotation(ConstructorProperties.class);
                    if(ctorProps != null) {
                         names = ctorProps.value();
                    } else { // we use parameter names
                        Parameter[] parms = descriptor.usableCtor.getParameters() ;
                        if(! parms[0].isNamePresent()) {
                            throw new IllegalArgumentException("no way to recompose the object (code not compiled with named parameters options) :" + value);
                        }
                        names = new String[parms.length] ;
                        for(int ix = 0 ; ix < parms.length ; ix++) {
                            names[ix] = parms[ix].getName() ;
                        }
                    }
                    Object[] args = new Object[names.length];
                    for (int ix = 0; ix < names.length; ix++) {
                        String name = names[ix];
                        //TODO inefficient and copy of KVList! change
                        for (KeyData keyData : codedData.listKV) {
                            if (keyData.getKey().equals(name)) {
                                args[ix] = decode(keyData.getRawValue());
                                break;
                            }
                        }
                    }
                    res = descriptor.usableCtor.newInstance(args);
                }
                return res;

            } else {
                //TODO throw exception
                throw new IllegalArgumentException("no way to recompose the object :" + value);
            }

        }
        return value;
    }

    /**
     * builds a key-value list from a coded composite data.
     * the key are assigned a composite name that start with the "rootName".
     *
     * @param rootName
     * @param codedData
     * @return
     */
    static List<KeyData> simpleKeyValueList(String rootName, CodedData codedData) {
        List<KeyData> resList = new ArrayList<>();
        // todo? bug if lisKV null?
        if(codedData.listKV == null) return resList ;
        for (KeyData keyVal : codedData.listKV) {
            Object val = keyVal.getRawValue();
            String key = keyVal.getKey();
            String longName;
            if (key.length() == 0) {
                longName = rootName;
            } else {
                longName = rootName + '/' + key;
            }
            if (val instanceof CodedData) {
                List<KeyData> contentList = asSimpleKeyValueList(longName, (CodedData) val);
                resList.addAll(contentList);

            } else {
                resList.add(new KeyData(longName, val));
            }
        }
        return resList;
    }

    /**
     * @param keyName
     * @param data
     * @return
     */
    public static List<KeyData> asSimpleKeyValueList(String keyName, Object data) {
        if (data instanceof CodedData) {
            return simpleKeyValueList(keyName, (CodedData) data);
        }
        List<KeyData> res = new ArrayList<>(1);
        res.add(new KeyData(keyName, data));
        return res;
    }


}
