package org.lsst.ccs.bus.messages;

import java.util.Arrays;
import org.lsst.ccs.command.BasicCommand;
import org.lsst.ccs.command.RawCommand;
import org.lsst.ccs.command.TokenizedCommand;

/**
 * Wrapper class to send a BasicCommand over the Command bus; the BasicCommand is either
 * provided as an object or created internally from a String representation: e.g. "doSomething 1 2".
 * 
 * The correlationId of this CommandBusMessage is create internally when the object is built.
 * 
 * @author LSST CCS Team
 * 
 */
public class CommandRequest extends CommandMessage {

    private final BasicCommand command;
    private String commandOriginatorId = null;

    /**
     * Create a BasicCommandMessage from a string representation of a command: e.g. "doSomething 1 2".
     * @param destination The destination of this command.
     * @param command     The string representation of a command. A BasicCommand object will be created internally.
     * @param args        The array of arguments to the command.
     * 
     */
    public CommandRequest(String destination, String command, Object ... args) {
        this(destination, (args != null && args.length > 0) ? new RawCommand(command, Arrays.asList(args)) : new TokenizedCommand(command));
    }
    

    /**
     * Create a BasicCommandMessage from a string representation of a command: e.g. "doSomething 1 2".
     * @param destination The destination of this command.
     * @param command     The string representation of a command. A BasicCommand object will be created internally.
     * 
     */
    public CommandRequest(String destination, String command) {
        this(destination, new TokenizedCommand(command));
    }
    
    /**
     * Create a BasicCommandMessage from an existing BasicCommand.
     * @param destination The destination of this command.
     * @param command     The BasicCommand object to be wrapper.
     */
    public CommandRequest(String destination, BasicCommand command) {
        super(destination);
        this.command = command;
    }
        
    
    public BasicCommand getBasicCommand() {
        return command;
    }
    

    @Override
    public String toString() {
        return "CommandRequest to : "+getDestination()+" "+getBasicCommand();
    }

    /**
     * Get the CommandOriginator Id for this CommandRequest.
     * 
     */
    public final void setCommandOriginatorId(String commandOriginatorId) {
        if ( this.commandOriginatorId != null ) {
            throw new RuntimeException("Method setCommandOriginatorId on CommandRequest must be invoked only once!!!");
        }
        this.commandOriginatorId = commandOriginatorId;
    }

    /**
     * Get the CommandOriginator Id for this CommandRequest.
     * 
     */
    public String getCommandOriginatorId() {
        return commandOriginatorId;
    }
}
