package org.lsst.ccs.bus;

import java.util.Iterator;

/**
 * Status with additional information.
 * <p/>
 * This is a model for the next version of DataStatus_Deprecated
 * <P/>
 * Instances of this class carry a set of data with a common <TT>dataTimeStamp</TT>.
 * Each data in this set (of type <TT>KVList</TT>) is an object with a name.
 * Each object is encoded in a <TT>KeyData</TT> structure.
 * <P/>
 * Note: keep in mind that each element in the list is of the form <TT>Key-VALUE</TT>
 * but the <TT>VALUE</TT>   can be itself burst into a list of
 * key-values (if it is a composite Object) (so do not confuse the keys!)
 * <P/>
 * So for example you can have
 * <UL>
 *     <LI/> a single primitive value contained as <TT>KVList{KeyData{"MyInteger",666}}</TT>
 *     <LI/> multiple primitive values contained as <TT>KVList{KeyData{"MyInteger",666}, KeyData{"MyString, "Hello"}}</TT>
 *     <LI/> single or multiple values that can be of any type (composite objects should have some specific properties: be <TT>Serializable</TT> or be "decomposable" in a list of key-value pairs) as:
 *     <TT>KVList{KeyData{"MyInteger", 666}, i
 *     KeyData{"MyObject", CodedData{"org.lsst.ccs.MyType", ,
 *     List{KeyData{"field1", 567}, KeyData{"field2", "world"}}}}}</TT>
 *     (in that case objects of type <TT>MyType</TT> have two fields: "field1" and "field2")
 *     <BR/> the important thing here is that the object will be read as <TT>
 *         List{"MyObject.field1"=567, "MyObject.field2"="world"}
 *     </TT>
 * </UL>
 * Methods encapsulate and simplify this analysis:
 * for reading <TT>KeyData</TT> objects see corresponding documentation.
 * <P/>
 * Instances of the current class can be "linked" together to be sent in a single message
 * across the buses. This is used when one wants to publish different values with different timestamps
 * in a same message.
 *
 * @author bamade
 */
// Date: 15/04/13

public class EncodedDataStatus extends Status implements Iterable<EncodedDataStatus> {


    private EncodedDataStatus nextStatus;
    private final KVList content;
    private final long dataTimeStamp;

    /**
     * creates a status with a list of Key-value pairs with the same timestamp
     *
     * @param timestampArg
     * @param content
     */
    public EncodedDataStatus(long timestampArg, KVList content) {
        this.dataTimeStamp = timestampArg;
        this.content = content;
    }

    /**
     * creates a status with a list of Key-value pairs with the same timestamp
     * which is generated at creation time.
     * @param content
     */
    public EncodedDataStatus(KVList content) {
        this(System.currentTimeMillis(), content);
    }

    /**
     * creates a status with a single key and value (the timestamp is also provided)
     * @param timestamp
     * @param key
     * @param value
     */
    public EncodedDataStatus(long timestamp, String key, Object value) {
        this(timestamp, new KVList(key, value));
    }

    /**
     * creates a status with a single key and value (the timestamp is generated)
     * @param key
     * @param value
     */
    public EncodedDataStatus(String key, Object value) {
        this(new KVList(key, value));
    }

    /**
     * adds another status at the end of the linked list of status sent in a single message
     * @param next
     */
    public void addStatus(EncodedDataStatus next) {
        EncodedDataStatus dataStatus2 = this;
        while (nextStatus != null) {
            dataStatus2 = nextStatus;
        }
        nextStatus = next;
    }

    /**
     * links another status next to this one (for sending a batch of status
     * in a single message)
     * @param next
     */
    public void setNext(EncodedDataStatus next) {
        nextStatus = next;
    }

    /**
     * helps iterating over a list of linked status message.
     * <P/>
     * So an idiom to read a message can be:
     * <PRE>
     *     EncodedDataStatus received = ..... ;
     *     for (EncodedDataStatus dataStatus : received) {
     *          KVList list = dataStatus.getContent();
     *          for (KeyData keyData : list) {
     *               .... dataStatus.getDataTimestamp() ....
     *              .... keyData.getKey() ....
     *              // one of those Three options
     *              .... keyData.getValue() .... // Optional
     *              .... keyData.getContentAsList() ....// may be empty
     *              .... keyData.getCrystallizedData() .... // Optional
     *          }
     *      }
     * </PRE>
     * @return
     */
    @Override
    public Iterator<EncodedDataStatus> iterator() {
        return new Iterator<EncodedDataStatus>() {

            EncodedDataStatus coming = EncodedDataStatus.this;

            @Override
            public boolean hasNext() {
                return coming != null;
            }

            @Override
            public EncodedDataStatus next() {
                EncodedDataStatus res = coming;
                coming = res.nextStatus;
                return res;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();

            }
        };
    }


    public KVList getContent() {
        return content;
    }

    public long getDataTimestamp() {
        return this.dataTimeStamp;
    }

    @Override
    public String toString() {
        return super.toString() + " {" +
                "content=" + content +
                ", dataTimeStamp=" + dataTimeStamp +
                "} ";
    }
}
