package org.lsst.ccs.bus;

import org.lsst.ccs.utilities.logging.Logger;

import java.io.Serializable;
import java.util.Arrays;

/**
 * A command sent on the CCS bus.
 * <p/>
 * Will be executed by the destination subsystem.
 *
 * @author Eric Aubourg
 */
public abstract class Command extends BusMessage implements CommandBusMessage {

    /**
     * to be used by subclasses for no args for commands
     */
    public static final Serializable[] NO_ARGS = new Serializable[0];
    private static final long serialVersionUID = 1488098756706412158L;
    protected static final Logger log = Logger.getLogger("org.lsst.ccs.bus.Command");

    @Override
    public String getMessageType() {
        return "lsst.command";
    }

    /* targeting */
    //@NotNull
    protected String destination = "*";
    //@NotNull (when lock manage is active)
    protected String key;
    /** The level of the <b>sender</b> of this command. */
    protected int level;
    /**
     * Correlation id, propagated from command to command
     */
    //@NotNull
    String correlId;
    // @NotNull (see  specific subclasses)
    protected String command;
    protected Object[] parameters = NO_ARGS;
    /**
     * Indicates that the command is a pure String to be parsed by the receiver.
     * In this case no arguments should be specified.
     */
    protected boolean unParsed = false;

    // from BusMessage : lacks origin, summary, priorityLevel, detailLevel
    // lacks  correlID, unparsed
    protected Command(String key, int level, String destination, String command, Object... parameterz) {
        this.level = level;
        this.destination = destination;
        this.command = command;
        this.parameters = parameterz;
    }

    /**
     * The command destination : subsystem name
     *
     * @return command destination
     */
    public String getDestination() {
        return destination;
    }

    /**
     * The command destination : subsystem name
     *
     * @param destination
     */
    public void setDestination(String destination) {
        this.destination = destination;
    }

    /* locking and arbitrating */
    /**
     * Key used for lock and arbitration. If set, incoming commands must have
     * the same key.
     *
     * @return the key
     */
    public String getKey() {
        return key;
    }

    /**
     * Key used for lock and arbitration. If set, incoming commands must have
     * the same key
     *
     * @param key
     */
    public void setKey(String key) {
        this.key = key;
    }

    /**
     * is this command a writing command that has to be protected ?
     * <BR/>
     * <B>Deprecated</B> this is defined in the Command annotation
     */
    @Deprecated
    public boolean isWrite() {
        return true;
    }

    /* can this command wait for a single-threaded ownerSubsystem to be ready */
    /**
     * <B>Deprecated</B> this is subject to re-evaluation of specs
     */
    @Deprecated
    protected boolean canWaitForReady = true;

    /**
     * can this command wait for a single-threaded ownerSubsystem to be ready ?
     *
     * @return a boolean
     */
    @Deprecated
    public boolean canWaitForReady() {
        return canWaitForReady;
    }

    /**
     * can this command wait for a single-threaded ownerSubsystem to be ready
     *
     * @param canWaitForReady
     */
    @Deprecated
    public void setCanWaitForReady(boolean canWaitForReady) {
        this.canWaitForReady = canWaitForReady;
    }

    /**
     * <B>Deprecated</B> this is defined in the Command annotation
     */
    @Deprecated
    protected boolean canRunInActiveMode = true;

    /**
     * Can this command be executed by a subsystem which is in "active" mode ?
     *
     * @return a boolean
     */
    @Deprecated
    public boolean canRunInActiveMode() {
        return canRunInActiveMode;
    }

    @Deprecated
    public void setCanRunInActiveMode(boolean canRunInActiveMode) {
        this.canRunInActiveMode = canRunInActiveMode;
    }

    @Override
    public String toString() {
        return getClass().getName() + " from " + origin + " to " + destination + "["
                + this.getCommand() + "(" + Arrays.toString(parameters) + ")";
    }

    public String getCorrelId() {
        return correlId;
    }

    /**
     * Correlation id, propagated from command to command. Users of this class
     * can set a specific correlation id, but otherwise one will be generated
     * automatically.
     */
    public void setCorrelId(String correlId) {
        this.correlId = correlId;
    }

    //TODO : 2 subclasses ...
    public boolean isUnParsed() {
        return unParsed;
    }

    public void setUnParsed(boolean unParsed) {
        this.unParsed = unParsed;
    }

    public String getCommand() {
        return command;
    }

    public Object[] getParameters() {
        return parameters;
    }

    public int getLevel() {
        return level;
    }

    @Deprecated
    public void setLevel(int level) {
        this.level = level;
    }
}
