package org.lsst.ccs.bus;

import org.lsst.ccs.bus.trending.Trendable;
import org.lsst.ccs.bus.trending.TrendingData;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;

/**
 * this class is to replace <TT>ValueNotification</TT>
 * which is temporarily kept for compatibility.
 * <p/>
 * Object that are not of a "well known type" (as of <TT>ObjectNType</TT> definition)
 * are "crystallized" for explicit deserialization by the receiving code.
 * They are also analyzed and if they are annotated with the <TT>Trending</TT> annotation
 * their content is split in as many key/primitiveValue as needed (this is a recursive process).
 *
 * @author bamade
 */
// Date: 05/04/13

public class DataValueNotification extends ValueNotification {
    private Map<String, ObjectNType> trendingDesc;

    public DataValueNotification(String name, Serializable object, long tStamp) {
        this.name = name;
        ObjectNType val = new ObjectNType(object);
        this.data = val;
        this.tStamp = tStamp;
        if (!val.isOfWellKnownType()) {
            createTrendingMap(name, object);
        }
    }

    public DataValueNotification(String name, Serializable object) {
        this(name, object.getClass(), object);
    }

    public DataValueNotification(String name, Class classOfData, Serializable object) {
        this.name = name;
        ObjectNType val = new ObjectNType(classOfData, object);
        this.data = val;
        this.tStamp = System.currentTimeMillis();
        if (!val.isOfWellKnownType()) {
            createTrendingMap(name, object);
        }
    }

    public DataValueNotification(String name, int val) {
        this(name, Integer.TYPE, val);
    }

    public DataValueNotification(String name, double val) {
        this(name, Double.TYPE, val);
    }

    public DataValueNotification(String name, float val) {
        this(name, Float.TYPE, val);
    }

    public DataValueNotification(String name, char val) {
        this(name, Character.TYPE, val);
    }

    // just for upward compatibility
    public DataValueNotification(ValueNotification notification) {
        //TODO : expand in readable code!
        this(notification.getName(),
                (notification.getData() instanceof Serializable) ? (Serializable) notification.getData() : String.valueOf(notification.getData()),
                notification.gettStamp());
    }

    //TODO: abstract this and enable for various annotations (for instance for Responses)
    private boolean createTrendingMap(String name, Serializable obj) {
        if (obj == null) return false;
        Trendable trendable = obj.getClass().getAnnotation(Trendable.class);
        if (trendable == null) {
            return false;
        }
        trendingDesc = new HashMap<String, ObjectNType>();
        updateTrendingMap(name, obj);
        return true;
    }

    //TODO: annotations on accessors is necessary
    private void updateTrendingMap(String name, Serializable obj) {
        Class clazz = obj.getClass(); // duplicate of previous call
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            TrendingData annotation = field.getAnnotation(TrendingData.class);
            if (annotation != null) {
                try {
                    field.setAccessible(true);
                    Object value = field.get(obj);
                    if (value == null) continue;
                    Class<?> fieldClass = field.getType();
                    String fieldName = annotation.key();
                    if ("".equals(fieldName)) {
                        fieldName = field.getName();
                    }
                    String newKey = name + "/" + fieldName;
                    //what if value is not serializable!
                    if (fieldClass.isPrimitive() || Serializable.class.isAssignableFrom(fieldClass)) {
                        ObjectNType objNtype = new ObjectNType(fieldClass, (Serializable) value);
                        if (objNtype.isOfWellKnownType()) {
                            trendingDesc.put(newKey, objNtype);
                        } else {
                            Trendable trendable = fieldClass.getAnnotation(Trendable.class);
                            if (trendable != null) {
                                updateTrendingMap(newKey, (Serializable) value);
                            } else {
                                //TODO? what to do?
                                updateTrendingMapWithString(newKey, value);
                            }
                        }
                    } else {
                        updateTrendingMapWithString(newKey, value);
                    }
                } catch (IllegalAccessException e) {
                    System.err.println("Non Accessible field!" + name + " " + obj);
                }
            }
        }
    }

    private void updateTrendingMapWithString(String name, Object obj) {
        trendingDesc.put(name, new ObjectNType(String.class, String.valueOf(obj)));
    }


    public boolean isOfWellKnownType() {
        if (data instanceof ObjectNType) {
            return ((ObjectNType) data).isOfWellKnownType();
        }
        return true;
    }

    public boolean isOfPrimitiveType() {
        if (data instanceof ObjectNType) {
            return ((ObjectNType) data).isOfPrimitiveType();
        }
        return false;
    }

    /**
     * Map of Trending key/values: the values are itself of type
     * <TT>ObjectNType</TT> so you can get the real type to be used.
     * @return
     */
    public Map<String, ObjectNType> getTrendingMap() {
        //TODO: should be readOnly
        return trendingDesc;
    }

    /**
     * to be used to retrieve the contained data.
     * An Exception is thrown if the contained Object cannot be deserialized
     *
     * @return
     * @throws ClassNotFoundException
     */
    public Object getRealData() throws ClassNotFoundException {
        if (data instanceof ObjectNType) {
            return ((ObjectNType) data).getData();
        }
        // should not happen
        return data;
    }

    /**
     * kept for compatibility reasons (use getRealData instead).
     *
     * @return
     */
    @Deprecated
    public Object getData() {
        try {
            return getRealData();
        } catch (ClassNotFoundException e) {
            if (data instanceof ObjectNType) {
                return ((ObjectNType) data).getRawData();
            }
            // should not happen
            return data;
        }
    }

    public ObjectNType getObjectNType() {
        return (ObjectNType) data;
    }

    @Override
    public String toString() {
        return "TrendingData(" + name + "=" + data + "@" + tStamp + ")";
    }
}
