package org.lsst.ccs.bootstrap;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Logger;

/**
 * Utility class for handling resources.
 *
 * @author turri
 */
class ResourcesUtils {

    private static final Properties defaultProperties = new Properties();

    /**
     * Set a default property to be load with the defaults in the process of
     * building properties files.
     * This is mainly meant to be used in a test environment.
     *
     * @param property The property name
     * @param value The value
     */
    static void setDefaultProperty(String property, String value) {
        defaultProperties.setProperty(property, value);
    }

    /**
     * Get a merged version of a given property file from the given
     * ResourcesTree. The resource directories are scanned from the bottom up.
     * Every time a property file for the given name is found, its properties
     * are loaded using the load method on the Properties object. So all the
     * properties are merged.
     *
     * @param tree The ResourcesTree in which property files are searched.
     * @param fileName The name of the property file to search for.
     * @return The Properties object with the merged properties.
     */
    static Properties getMergedPropertyFile(ResourcesTree tree, String fileName) {
        return getMergedPropertyFile(tree, fileName, false);
    }

    /**
     * Get a merged version of a given property file from the given
     * ResourcesTree. The resource directories are scanned from the bottom up.
     * Every time a property file for the given name is found, its properties
     * are loaded using the load method on the Properties object. So all the
     * properties are merged.
     *
     * @param tree The ResourcesTree in which property files are searched.
     * @param fileName The name of the property file to search for.
     * @param useSystem If true the System Properties will be the parent of the
     * returned Properties object.
     * @return The Properties object with the merged properties.
     */
    static Properties getMergedPropertyFile(ResourcesTree tree, String fileName, boolean useSystem) {
        return getMergedProperties(tree, new String[]{fileName}, useSystem, null);

    }

    static Properties getMergedProperties(ResourcesTree tree, String[] properties, boolean useSystem, ResourcesTreeProperties parentProps) {

        //First load the System properties is useSystem is set to true
        ResourcesTreeProperties props = parentProps;
        if (useSystem) {
            if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                System.out.println("*** Adding System Properties to the properties chain");
            }
            props = new ResourcesTreeProperties("SystemProperties", null, props);
            props.putAll(System.getProperties());
        }

        //Then load the properties from ccsDefaults.properties in the distribution resource directory
        try {
            InputStream defaultsIn = Bootstrap.getLoaderClass().getResourceAsStream("/ccsDefaults.properties");
            if (defaultsIn != null) {
                if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                    System.out.println("*** Adding Properties " + BootstrapUtils.getDistributionResourcesDirectory() + "ccsDefaults.properties to properties chain");
                }
                Properties parent = props;
                props = new ResourcesTreeProperties("ccsDefaults", BootstrapUtils.getDistributionResourcesDirectory(), parent);
                props.load(defaultsIn);
                String applicationName = Bootstrap.getBootstrapApplication();
                props.put("org.lsst.ccs.application.name", applicationName == null ? "" : applicationName);
            }
        } catch (IOException ioe) {
            throw new RuntimeException(ioe);
        }

        Properties dProperties = System.getProperties();
        for (String key : dProperties.stringPropertyNames()) {
            if (key.startsWith("bootstrap.default.")) {
                defaultProperties.put(key.replace("bootstrap.default.", ""), dProperties.getProperty(key));
            }
        }

        //Add the internal defaults
        if (!defaultProperties.isEmpty()) {
            props = new ResourcesTreeProperties("defaults", null, props);
            props.putAll(defaultProperties);
        }

        List<ResourceDirectory> resourceList = tree.getResourceDirectoryList();
        ListIterator<ResourceDirectory> reverseIterator = resourceList.listIterator(resourceList.size());
        while (reverseIterator.hasPrevious()) {
            ResourceDirectory dir = reverseIterator.previous();
            for (String propertyFileName : properties) {
                if (propertyFileName != null) {
                    if (!propertyFileName.endsWith(".properties")) {
                        propertyFileName += ".properties";
                    }

                    // Check if the property file exists either in its fully qualified path, or at the root
                    File propertyFile = new File(dir.getResouceDirectoryPath(), propertyFileName);
                    if (!propertyFile.exists()) {
                        if (propertyFileName.contains(BootstrapUtils.FILE_SEPARATOR)) {
                            propertyFileName = propertyFileName.substring(propertyFileName.lastIndexOf(BootstrapUtils.FILE_SEPARATOR) + 1);
                            propertyFile = new File(dir.getResouceDirectoryPath(), propertyFileName);
                        }
                    }

                    if (propertyFile.exists()) {
                        FileInputStream in = null;
                        try {
                            in = new FileInputStream(propertyFile);
                            Properties parent = props;
                            props = new ResourcesTreeProperties(propertyFileName, dir.getResouceDirectoryPath(), parent);
                            props.load(in);
                            if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                                System.out.println("*** Adding Properties " + dir.getResouceDirectoryPath() + propertyFileName + " to properties chain");
                            }
                        } catch (IOException ioe) {
                            throw new RuntimeException(ioe);
                        } finally {
                            if (in != null) {
                                try {
                                    in.close();
                                } catch (IOException e) {
                                    throw new RuntimeException(e);
                                }
                            }
                        }
                    }
                }
            }
        }        
        return props;
    }

    static void printProperties(Properties props) {
        printProperties(props,null,null);
    }
    
    static void printProperties(Properties props, StringBuilder sb, Logger logger) {
        if ( sb == null ) {
            sb = new StringBuilder();
        }
        sb.append("*****************\n");
        if (props instanceof ResourcesTreeProperties) {
            ResourcesTreeProperties p = (ResourcesTreeProperties) props;
            String output = "Properties from " + p.getPropertyFileName();
            if (p.getResourceDirectory() != null) {
                output += " in resource directory " + p.getResourceDirectory();
            }
            sb.append(output).append("\n");
        } else {
            sb.append("External System properties");
        }
        for (Object key : props.keySet()) {
            sb.append("\t").append(key).append(" = ").append(props.getProperty((String) key)).append("\n");
        }
        if (props instanceof ResourcesTreeProperties) {
            ResourcesTreeProperties p = (ResourcesTreeProperties) props;
            if (p.hasParent()) {
                printProperties(p.getParent(),sb,logger);
            }
        }        
        if ( logger == null ) {
            System.out.println(sb);
        } else {
            logger.fine(sb.toString());
        }
    }
    

    static void loadKeySetForProperties(Properties props, Set<Object> set) {
        Set<Object> propsKey = props.keySet();
        for (Object obj : propsKey) {
            if (!set.contains(obj)) {
                set.add(obj);
            }
        }
        if (props instanceof ResourcesTreeProperties && ((ResourcesTreeProperties) props).getParent() != null) {
            loadKeySetForProperties(((ResourcesTreeProperties) props).getParent(), set);
        }
    }

    static InputStream getResourceFromResourceTree(ResourcesTree tree, String resourceName, Logger logger) throws FileNotFoundException {
        List<ResourceDirectory> resourceList = tree.getResourceDirectoryList();

        for (ResourceDirectory dir : resourceList) {
            File resourceFile = new File(dir.getResouceDirectoryPath() + resourceName);
            if (resourceFile.exists()) {
                if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                    System.out.println("*** Found resource " + dir.getResouceDirectoryPath() + resourceName);
                }
                if ( logger != null ) {
                    logger.fine("Found resource "+dir.getResouceDirectoryPath()+resourceName);
                }
                return new FileInputStream(resourceFile);
            }
        }
        return null;
    }

}
