package org.lsst.ccs.bootstrap;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;
import java.util.Set;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import org.apache.commons.cli.Options;
import static org.lsst.ccs.bootstrap.ResourcesUtils.printProperties;

/**
 * Utility methods for loading Bootstrap Resources.
 * <p>
 * The utility methods provided in this class are meant to be used throughout the CCS environment
 * to load resources in the Bootstrap environment.
 * <p>
 * For a description on how to control the Bootstrap environment please refer to the
 * following <a href="https://confluence.slac.stanford.edu/x/-upFC">manual</a>.
 *
 * @author The LSST CCS Team
 */
public abstract class BootstrapResourceUtils {

    private static final Logger logger = Logger.getLogger("org.lsst.ccs.bootstrap");
    
    public static List<String>[] separateArgumentsForOptions(String[] args, Options options) {
        
        List<String>[] result = new ArrayList[2];
        
        //Check for arguments that start with -D and don't have value associated with it.
        //i.e. arguments of the form: -D some.prop
        //Also detect if there are any command line options that don't belong to 
        //the bootstrap. If any is found they are stored to be passed to the main application.
        List<String> argumentsToParse = new ArrayList<>();
        List<String> argumentsLeftOver = new ArrayList<>();
        boolean skipNext = false;
        for (int i = 0; i < args.length; i++) {
            if ( skipNext ) {
                skipNext = false;
                continue;
            }
            String arg = args[i];
            if (arg.startsWith("-D")) {
                // Check if it's a "-D", i.e. there is a space "-D prop=value"
                int propertyStrIndex = arg.equals("-D") ? i + 1 : i;
                String propertyStr = args[propertyStrIndex].replace("-D", "");
                // Check if there is a property separator, either = or :
                if (!propertyStr.contains("=") /*&& ! propertyStr.contains(":")*/) {
                    args[propertyStrIndex] = args[propertyStrIndex].replace(propertyStr, propertyStr + "=");
                }
                if (arg.equals("-D")) {
                    argumentsToParse.add(arg);
                } else {
                    argumentsLeftOver.add(args[i]);
                }
            } else {
                if ( arg.startsWith("-") && ! options.hasOption(arg) ) {
                    //This is an option that does not belong to the bootstrap.
                    argumentsLeftOver.add(arg);
                    //Check if the option is followed by another argument
                    if ( i < args.length -1 ) {
                        String tmpArg = args[i+1];
                        if ( ! tmpArg.startsWith("-") ) {
                            argumentsLeftOver.add(tmpArg);
                            skipNext = true;
                        }
                    }
                } else {
                    argumentsToParse.add(arg);
                    argumentsLeftOver.add(arg);
                }
                
            }
        }
        
        result[0] = new ArrayList<>(argumentsToParse);
        result[1] = new ArrayList<>(argumentsLeftOver);
        return result;
    } 

    /**
     * Return the Bootstrap System Properties object.
     * <p>
     * This function builds a Properties object by searching and chaining properties
     * according to the following search criteria
     * <ul>
     * <li><b>System Properties</b>: System Properties are the first Properties
     * to be added to the merge chain.</li>
     * <li><b>Bootstrap Resources search</b> : The Bootstrap Resources directories
     * (defined by the environment variable CCS_RESOURCE_PATH) are searched in the
     * reversed order. In each directory properties are merged by searching for
     * application and global properties files (these files are loaded from the
     * file system). </li>
     * <li><b>Command line </b>: Properties specified from the command line with
     * the -D option are added to the top of the Properties chain.</li>
     * </ul>
     *
     * @return The merged Properties object.
     */
    public static Properties getBootstrapSystemProperties() {
        if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
            System.out.println("*** Getting Bootstrap System Properties");
        }
        return buildBootstrapProperties(null, null, true);
    }

    /**
     * Utility function to find the top most property file in the user provided
     * resource directories (as specified in the CCS_RESOURCE_PATH environment
     * variable). If the CCS_RESOURCE_PATH environment variable is not specified
     * null will be returned. If specified the first property file will be returned
     * scanning the resource directories in CCS_RESOURCE_PATH left to right.
     *
     * @param propertyFileName The name of the property file to get
     * @return The full path of the property file.
     *
     */
    public static String getPathOfPropertiesFileInUserResourceDirectories(String propertyFileName) {

        if (!propertyFileName.endsWith(".properties")) {
            propertyFileName += ".properties";
        }

        return getPathOfResourceInUserResourceDirectories(propertyFileName);
    }

    public static String getPathOfResourceInUserResourceDirectories(String fullFileName) {

        ResourcesTree tree = BootstrapUtils.getBootstrapResourcesTree();

        ListIterator<ResourceDirectory> resourceDirs = tree.getResourceDirectoryList().listIterator();
        while (resourceDirs.hasNext()) {
            ResourceDirectory resourceDir = resourceDirs.next();
            if (!resourceDir.isDistributionDir()) {
                File resourceFile = new File(resourceDir.getResouceDirectoryPath() + fullFileName);
                if (resourceFile.exists()) {
                    return resourceDir.getResouceDirectoryPath() + fullFileName;
                }
            }
        }

        return null;
    }

    /**
     * Utility function to find the URL of a resource by name.
     * The first instance of this resource will be loaded scanning all
     * the CCS resource paths from the top most to the lower one (the actual
     * resource directory in the distribution).
     *
     * @param resourceName The name of the resource get
     * @return The full path of the resource.
     *
     */
    public static URL getResourceURL(String resourceName) throws MalformedURLException {

        ResourcesTree tree = BootstrapUtils.getBootstrapResourcesTree();

        ListIterator<ResourceDirectory> resourceDirs = tree.getResourceDirectoryList().listIterator();
        while (resourceDirs.hasNext()) {
            ResourceDirectory resourceDir = resourceDirs.next();
            File resourceFile = new File(resourceDir.getResouceDirectoryPath() + resourceName);
            if (resourceFile.exists()) {
                return resourceFile.toURI().toURL();
            }
        }

        if ( ! resourceName.startsWith("/") ) {
            resourceName = "/" + resourceName;
        }

        URL url = BootstrapUtils.class.getResource(resourceName);
        if ( url != null ) {
            logger.fine("Found resource loacation "+url);
            return url;
        }

        return null;
    }

    /**
     * Utility function to get the top User defined Resource Directory.
     * This is the left-most directory defined in environment variable CCS_RESOURCE_PATH.
     * If not defined, null is returned.
     *
     * @return the top user defined Resource directory. Null if not defined.
     */
    public static String getTopUserResourceDirectory() {
        ResourcesTree tree = BootstrapUtils.getBootstrapResourcesTree();

        ListIterator<ResourceDirectory> resourceDirs = tree.getResourceDirectoryList().listIterator();
        while (resourceDirs.hasNext()) {
            ResourceDirectory resourceDir = resourceDirs.next();
            if (!resourceDir.isDistributionDir()) {
                return resourceDir.getResouceDirectoryPath();
            }
        }
        return null;
    }

    /**
     * Utility function to fetch Properties in the Bootstrap environment.
     * <p>
     * This function builds a Properties object by searching and chaining properties
     * according to the following search criteria
     * <ul>
     * <li><b>Classpath search</b>: getResourceAsStream is invoked on the provided class.
     * The method is first invoked for fileName. If not input stream is found it is then invoked for
     * "/"+fileName</li>
     * <li><b>System Properties</b>: if the parameter includeSystem is set to true, System
     * Properties are added to the chain</li>
     * <li><b>Bootstrap Resources search</b>:
     * The Bootstrap Resources directories (defined by the environment variable
     * CCS_RESOURCE_PATH) are searched in the reversed order. In each directory
     * properties are merged by searching for application, global and topic
     * (defined by fileName) properties files (these files are loaded from the
     * file system). If fileName is a path to a properties file, two searches
     * are performed for each resource directory: the first one is for the fully
     * qualified path; if this yields no result, the properties file is then
     * searched at the root of each resource directory.</li>
     * <li><b>Command line</b>:
     * Properties specified from the command line with the -D option are added
     * to the top of the Properties chain.</li>
     * </ul>
     *
     * @param fileName The name of the properties file to load. This can be a path.
     * @param clazz The class on which getResourceAsStream is invoked for the classpath search. If null BootstrapResourceUtils.class is used.
     * @return The merged Properties object.
     *
     */
    public static Properties getBootstrapProperties(String fileName, Class clazz) {
        if ( ! fileName.endsWith(".properties") ) {
            fileName += ".properties";
        }
        if (clazz == null) {
            clazz = Bootstrap.getLoaderClass();
        }
        if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
            System.out.println("*** Looking for Properties " + fileName);
        }
        InputStream in = getResourceFromClassPath(clazz, fileName);
        return buildBootstrapProperties(in, fileName, false);
    }

    /**
     * Utility function to fetch Properties in the Bootstrap environment.
     * <p>
     * This function is similar to {@link #getBootstrapProperties(java.lang.String, boolean, java.lang.Class) }.
     * In this case the Class object is set to null and it defaults to BootstrapResourceUtils.class.
     *
     * @see #getBootstrapProperties(java.lang.String, boolean, java.lang.Class)
     * @param fileName The name of the properties file to load. This can be a path.
     * @return The merged Properties object.
     *
     */
    public static Properties getBootstrapProperties(String fileName) {
        return getBootstrapProperties(fileName, Bootstrap.getLoaderClass());
    }

    private static Properties buildBootstrapProperties(InputStream in, String fileName, boolean includeSystem) {
        ResourcesTreeProperties props = null;

        if (in != null && fileName != null) {
            props = new ResourcesTreeProperties("ClassPath " + fileName, null, null);
            try {
                props.load(in);
            } catch (IOException ioe) {
                throw new RuntimeException("Problem loading properties from classpath", ioe);
            } finally {
                try {
                    in.close();
                } catch (IOException ioe) {
                    throw new RuntimeException("Could not close stream.",ioe);
                }
            }
        }

        String applicationLevel = Bootstrap.getBootstrapApplication();
        String topicLevel = fileName;

        Properties appProperties = new Properties();
        try {
            InputStream is = getBootstrapResource(applicationLevel+".properties");
            if ( is != null ) {
                appProperties.load( is );
                try {
                    is.close();
                } catch (IOException ioe) {
                    throw new RuntimeException("Could not close stream.",ioe);
                }
            }
            is = getBootstrapResource(applicationLevel+".app");
            if ( is != null ) {
                appProperties.load( is );
                try {
                    is.close();
                } catch (IOException ioe) {
                    throw new RuntimeException("Could not close stream.",ioe);
                }
            }
        } catch (Exception e) {    
        }
        String[] extensions = appProperties.getProperty("org.lsst.ccs.application.extends","").split(",");
        ArrayList<String> appsToLoad = new ArrayList<>();
        ArrayList<String> extensionsToLoad = new ArrayList<>();
        for ( String extension : extensions ) {
            appsToLoad.add(extension.trim());
            appsToLoad.add(extension.trim());
            extensionsToLoad.add("app");
            extensionsToLoad.add("properties");
        }
        Collections.reverse(appsToLoad);
        appsToLoad.add(topicLevel);
        extensionsToLoad.add("properties");
        appsToLoad.add("ccsGlobal");
        extensionsToLoad.add("properties");
        appsToLoad.add(applicationLevel);
        extensionsToLoad.add("app");
        appsToLoad.add(applicationLevel);
        extensionsToLoad.add("properties");
        
        props = (ResourcesTreeProperties) ResourcesUtils.getMergedProperties(BootstrapUtils.getBootstrapResourcesTree(), appsToLoad.toArray(new String[0]), includeSystem, props,extensionsToLoad.toArray(new String[0]));
        if (!Bootstrap.getCmdLineProperties().isEmpty()) {
            if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                System.out.println("*** Adding command line properties to properties chain.");
            }
            ResourcesTreeProperties cmdLinePros = new ResourcesTreeProperties("Command Line Properties", null, props);
            cmdLinePros.putAll(Bootstrap.getCmdLineProperties());
            props = cmdLinePros;
        }

        if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
            ResourcesUtils.printProperties(props);
        }
        
        StringBuilder sb = new StringBuilder();
        sb.append("** Found properties "+fileName);
        printProperties(props, sb, logger);        
        return props;

    }

    /**
     * Get an InputStream corresponding to the Bootstrap Properties object.
     * <p>
     * This function is similar to {@link #getBootstrapProperties(java.lang.String, boolean, java.lang.Class) }.
     * Rather than a Properties object it returns the InputStream corresponsing to
     * that Properties object. This is done by invoking {@Code java.util.Properties#store(java.io.OutputStream, java.lang.String) }
     * on a ByteArrayOutputStream and then returning an ByteArrayInputStream.
     * <p>
     *
     * @see BootstrapResourceUtils#getBootstrapProperties(java.lang.String, boolean, java.lang.Class)
     * @param fileName The name of the properties file to load. This can be a path.
     * @param clazz The class on which getResourceAsStream is invoked for the classpath search. If null BootstrapResourceUtils.class is used.
     * @return The input stream for the specified properties file. Null if the properties file does not exist.
     */
    public static InputStream getBootstrapPropertiesInputStream(String fileName, Class clazz) {
        Properties props = getBootstrapProperties(fileName, clazz);
        if (props == null) {
            return null;
        }

        Properties outProps = props;
        if (props instanceof ResourcesTreeProperties) {
            outProps = new Properties();
            ((ResourcesTreeProperties) props).copyProperties(outProps);
        }

        InputStream is = null;
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        try {
            outProps.store(output, null);
            is = new ByteArrayInputStream(output.toByteArray());
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }

        return is;
    }

    /**
     * Utility functions to look for resources (excluding Properties) in the Bootstrap environment.
     * <p>
     * This function will search for the desired resource following the search criteria described below.
     * It will return an InputStream for the first resource found.
     * <p>
     * Search Criteria:
     * <ul>
     * <li><b>Bootstrap Resources search</b>: The Bootstrap Resources directories (defined by the
     * environment variable CCS_RESOURCE_PATH) are searched in the defined
     * order. If fileName is a path to a resource, two searches are performed
     * for each resource directory: the first one is for the fully qualified
     * path; if this yields no result, the resource file is then searched at the
     * root of each resource directory.</li>
     * <li><b>Classpath search</b>:
     * getResourceAsStrean is invoked on the provided clazz. The method is first invoked for
     * fileName. If not input stream is found it is then invoked for "/"+fileName</li>
     * </ul>
     * <p>
     * For Properties refer to {@link #getBootstrapProperties(java.lang.String, boolean,java.lang.Class).
     *
     * @param resourceName The name of the resource. This can be a path.
     * @param clazz The class on which getResourceAsStream is invoked for the
     * classpath search. If null BootstrapResourceUtils.class is used.
     * @return The InputStream of the first found resource. It will return null if the resource is not found.
     *
     */
    public static InputStream getBootstrapResource(String resourceName, Class clazz) {
        if (clazz == null) {
            clazz = Bootstrap.getLoaderClass();
        }
        if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
            System.out.println("*** Looking for resource " + resourceName + " for Class " + clazz);
        }

        InputStream in = null;

        try {
            in = ResourcesUtils.getResourceFromResourceTree(BootstrapUtils.getBootstrapResourcesTree(), resourceName, logger);
        } catch (FileNotFoundException fnfe) {
            throw new RuntimeException(fnfe);
        }
        
        if (in != null) {
            return in;
        } else {
            return getResourceFromClassPath(clazz, resourceName);
        }

    }

    /**
     * Utility functions to look for resources (excluding Properties) in the Bootstrap environment.
     * <p>
     * Internally it invokes {@link getBootstrapResource(java.lang.String,java.lang.Class)} with Class
     * set to BootstrapResourceUtils.class.
     *
     * @see #getBootstrapResource(java.lang.String, java.lang.Class)
     * @param resourceName The name of the resource. This can be a path.
     * @return The InputStream of the first found resource.
     *
     */
    public static InputStream getBootstrapResource(String resourceName) {
        return getBootstrapResource(resourceName, Bootstrap.getLoaderClass());
    }

    /**
     * Utility method to fetch all the keys in a properties file, including its parent, to loop over them.
     * <p>
     * This method keeps into account if a Properties object has a parent and goes down
     * the parent chain to fetch all the keys.
     *
     * @param props The Properties object
     * @return The Set<Object> of all the keys in the Properties object.
     */
    public static Set<Object> getAllKeysInProperties(Properties props) {
        Set<Object> keySet = new HashSet<>();
        if (props != null) {
            ResourcesUtils.loadKeySetForProperties(props, keySet);
        }
        return keySet;
    }

    
    /**
     * Get the ordered list of resource directories.
     *
     * The built in default is: - Distribution resources directory:
     * distribution/etc/ - System resources directory: /etc/ccs/ - User
     * resources directory: ~/.ccs/etc/
     *
     * By setting the environment variable CCS_RESOURCE_PATH, the developer can
     * change which resource directories are searched and in which order.
     *
     * When the CCS_RESOURCE_PATH environment variable is set, the distribution
     * resource directory will be still searched first, before any of the
     * directories provided by the user.
     *
     * This list is built once and its result is cached.
     *
     * @return The list of ordered directories used to search for resources.
     *
     */
    public static List<String> getOrderedListOfResourceDirectories() {
        return BootstrapUtils.getOrderedListOfResourceDirectories();
    }
    
    private static InputStream getResourceFromClassPath(Class clazz, String resourceName) {
        InputStream in = clazz.getResourceAsStream(resourceName);
        if (in == null && !resourceName.startsWith("/")) {
            resourceName = "/" + resourceName;
            in = clazz.getResourceAsStream(resourceName);
        }
        if ((Bootstrap.verbose() && in != null) && !Bootstrap.isQuiet()) {
            System.out.println("*** Found resource " + resourceName + " in classpath for Class " + clazz);
        }
        if ( in != null ) {
            logger.fine("Found resource "+resourceName+" from classpath");
        }
        return in;
    }

    /**
     * Find all resources that match the provided regular expression pattern.
     * All resource directories will be searched.
     * @param pattern The regular expression pattern to be used in the search.
     * @return The Set of the resource file names matching the provided pattern.
     * 
     */
    public static Set<String> findMatchingResources(String pattern) {
        Set<String> result = new HashSet<>();
        Pattern p = Pattern.compile(pattern);
        ResourcesTree tree = BootstrapUtils.getBootstrapResourcesTree();
        for ( ResourceDirectory dir : tree.getResourceDirectoryList() ) {
            for ( String fileName : dir.findMatchingResources(p) ) {
                result.add(fileName);
            }
        }
        return result;
    }
    
    /**
     * Get the version information written in the jar file manifest for a given
     * Class.
     * @param clazz The class for which the versions are to be fetched.
     * @return The properties containing the version information. An empty 
     *         properties object if the jar file cannot be found.
     */
    public static Properties getManifestVersionsForClass(Class clazz) {
        try {
            URL classUrl = clazz.getProtectionDomain().getCodeSource().getLocation();
            File f = new File(classUrl.toURI());
            return scanManifestForVersions(f);
        } catch (Exception e) {
            return new Properties();
        }
    }

    /**
     * Get the version information written in the jar file manifest for the given distribution.
     * @return The properties containing the version information.
     */
    public static Properties getManifestVersionsForDistribution() {
        try {
            String applicationMainJar = Bootstrap.getDistributionMainJar();
            if (applicationMainJar != null) {
                File jar = new File(applicationMainJar);
                return scanManifestForVersions(jar);
            }
        } catch (Exception e) {
            
        }
        return new Properties();
    }
   
    protected static Properties scanManifestForVersions(File f) {
        Properties props = new Properties();
        try (JarFile jarFile = new JarFile(f) ) {
            Manifest manifest = jarFile.getManifest();
            props.setProperty("org.lsst.ccs.project", manifest.getMainAttributes().getValue("CCS-Project"));
            props.setProperty("org.lsst.ccs.project.version", manifest.getMainAttributes().getValue("Implementation-Version"));
            props.setProperty("org.lsst.ccs.jenkins.build.number", manifest.getMainAttributes().getValue("CCS-Jenkins-Build-Number"));
            props.setProperty("org.lsst.ccs.jenkins.build.id", manifest.getMainAttributes().getValue("CCS-Jenkins-Build-Id"));
            props.setProperty("org.lsst.ccs.jenkins.build.url", manifest.getMainAttributes().getValue("CCS-Jenkins-Build-Url"));
            props.setProperty("org.lsst.ccs.jenkins.build.jdk", manifest.getMainAttributes().getValue("Build-Jdk"));
            props.setProperty("org.lsst.ccs.source.code.url", manifest.getMainAttributes().getValue("CCS-Source-Code-Url"));
            props.setProperty("org.lsst.ccs.source.code.revision", manifest.getMainAttributes().getValue("CCS-Source-Code-Revision"));
        } catch (Exception ioe) {
//            ioe.printStackTrace();
        } 
        return props;
    }

    

}
