package org.lsst.ccs.bootstrap;

import java.io.File;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utility methods for the Bootstrap application.
 *
 * @author turri
 * 
 * This class will no longer be public
 */
class BootstrapUtils {

    protected static final String FILE_SEPARATOR = System.getProperty("file.separator");
    protected static final String PATH_SEPARATOR = System.getProperty("path.separator");
    protected static final String DISTRIBUTION_JARS_DIRECTORY = "share" + FILE_SEPARATOR + "java" + FILE_SEPARATOR;
    private static final String DISTRIBUTION_RESOURCES_DIRECTORY = "etc" + FILE_SEPARATOR;
    private static final String DISTRIBUTION_LIB_DIRECTORY = "lib" + (System.getProperty("sun.arch.data.model").equals("32") ? "" : System.getProperty("sun.arch.data.model")) + FILE_SEPARATOR;
    private static final String DISTRIBUTION_JNI_DIRECTORY = DISTRIBUTION_LIB_DIRECTORY + "jni" + FILE_SEPARATOR;
    static final String APPLICATION_MAINCLASS_PROPERTY = "org.lsst.ccs.application.mainClass";
    static final String APPLICATION_EXTENDS_PROPERTY = "org.lsst.ccs.application.extends";
    static final String OBSOLETE_MAINJAR_PROPERTY = "org.lsst.ccs.application.mainJar";
    private volatile static List<String> listOfApplications = null;

    // Pattern to substitute environment variable in user provided properties
    private static final String envVar_str_pattern = ".*(\\$\\[env\\.(.*)\\]).*";
    private static final Pattern envVar_pattern = Pattern.compile(envVar_str_pattern);

    /**
     * Environment variable that defines the resource path.
     */
    private final static String CCS_RESOURCE_PATH_ENV_VAR = "CCS_RESOURCE_PATH";
    private final static String CCS_DISTRIBUTION_PATH_ENV_VAR = "CCS_DISTRIBUTION_PATH";
    private volatile static String ccsDistributionRootDirectory = null;
    private volatile static ResourcesTree resourcesTree = null;

    private static final Logger LOG = Logger.getLogger(BootstrapUtils.class.getName());

    protected static void reset() {
        resourcesTree = null;
        ccsDistributionRootDirectory = null;
        listOfApplications = null;
    }

    /**
     * Get the ResourceTree for this Bootstrap environment. It contains all the
     * resources defined in the resource directory list.
     *
     * @return
     */
    synchronized static ResourcesTree getBootstrapResourcesTree() {
        if (resourcesTree == null) {
            resourcesTree = new ResourcesTree();
            List<String> resourceDirectories = getOrderedListOfResourceDirectories();
            List<String> additionalResourceDirectories = extractDirectoriesFromPath(getUserProvidedResourceDirectories(), null, true);

            for (String resourceDirectory : resourceDirectories) {
                File resourceDir = new File(resourceDirectory);
                if (resourceDir.exists()) {
                    if (additionalResourceDirectories.contains(resourceDirectory)) {
                        resourcesTree.addUserResourceDirectory(resourceDirectory);
                    } else {
                        resourcesTree.addDistributionResourceDirectory(resourceDirectory);
                    }
                } 
             }
        }
        return resourcesTree;
    }

    static String getUserProvidedResourceDirectories() {
        String resourceProperty = "org.lsst." + CCS_RESOURCE_PATH_ENV_VAR.replace("_", ".").toLowerCase();
        String userProvidedResourceDirectories = System.getProperty(resourceProperty);
        return userProvidedResourceDirectories == null ? System.getenv(CCS_RESOURCE_PATH_ENV_VAR) : userProvidedResourceDirectories;
    }

    static String getUserProvidedDistributionDirectories() {
        String distributionProperty = "org.lsst." + CCS_DISTRIBUTION_PATH_ENV_VAR.replace("_", ".").toLowerCase();
        String userProvidedDistributionDirectories = System.getProperty(distributionProperty);
        return userProvidedDistributionDirectories == null ? System.getenv(CCS_DISTRIBUTION_PATH_ENV_VAR) : userProvidedDistributionDirectories;
    }

    /**
     * Get the ordered list of resource directories.
     *
     * The built in default is: - Distribution resources directory:
     * distribution/etc/ - System resources directory: /etc/ccs/ - User
     * resources directory: ~/.ccs/etc/
     *
     * By setting the environment variable CCS_RESOURCE_PATH, the developer can
     * change which resource directories are searched and in which order.
     *
     * When the CCS_RESOURCE_PATH environment variable is set, the distribution
     * resource directory will be still searched first, before any of the
     * directories provided by the user.
     *
     * This list is built once and its result is cached.
     *
     * @return The list of ordered directories used to search for resources.
     *
     */
    static List<String> getOrderedListOfResourceDirectories() {
        String userProvidedResourceDirs = getUserProvidedResourceDirectories();
        String userProvidedDistributionDirs = getUserProvidedDistributionDirectories();
        return getOrderedListOfResourceDirectories(userProvidedResourceDirs, userProvidedDistributionDirs, true);
    }

    /**
     * This method should do all the work of setting up the resource directories
     * as it is used in by the tests.
     *
     * @param userProvidedResourceDirs This resembles the environment variable
     * to set the list of resource directories.
     * @param userProvidedDistributions This resembles the environment variable
     * to set the list of additional distributions.
     * @return The list of ordered directories to search for resources.
     *
     */
    static List<String> getOrderedListOfResourceDirectories(String userProvidedResourceDirs, String userProvidedDistributions, boolean checkExistance) {

        String distributionResourceDirectoriy = getDistributionResourcesDirectory();

        List<String> additionaDistributionResourceDirectories = extractDirectoriesFromPath(userProvidedDistributions, DISTRIBUTION_RESOURCES_DIRECTORY, checkExistance);

        List<String> additionalResourceDirectories = extractDirectoriesFromPath(userProvidedResourceDirs, null, checkExistance);

        List<String> orderedListOfResourceDirectories = new ArrayList<>();

        // First add the additional resource directories
        for (String dir : additionalResourceDirectories) {
            orderedListOfResourceDirectories.add(dir);
        }

        // Then add the main distribution directory
        orderedListOfResourceDirectories.add(distributionResourceDirectoriy);

        // Finally add any additional distribution directory
        for (String dir : additionaDistributionResourceDirectories) {
            orderedListOfResourceDirectories.add(dir);
        }

        return orderedListOfResourceDirectories;
    }

    static List<String> extractDirectoriesFromPath(String directoryPath, String appendDirectory, boolean checkExistance) {
        return extractDirectoriesFromPath(BootstrapUtils.getCCSDistributionRootDirectory(), directoryPath, appendDirectory, checkExistance);
    }

    static List<String> extractDirectoriesFromPath(String root, String directoryPath, String appendDirectory, boolean checkExistance) {

        ArrayList<String> extractedDirectories = new ArrayList<>();

        if (directoryPath != null && !"".equals(directoryPath)) {
            StringTokenizer dirsToken = new StringTokenizer(directoryPath, PATH_SEPARATOR);
            while (dirsToken.hasMoreTokens()) {
                String resourceDir = dirsToken.nextToken().trim();
                if (!resourceDir.endsWith(FILE_SEPARATOR)) {
                    resourceDir += FILE_SEPARATOR;
                }
                if (appendDirectory != null && !"".equals(appendDirectory)) {
                    if (!appendDirectory.endsWith(FILE_SEPARATOR)) {
                        appendDirectory += FILE_SEPARATOR;
                    }
                    resourceDir += appendDirectory;
                }

                //The root for relative paths is the root of the CCS distribution.
                Path p = Paths.get(root);
                resourceDir = resourceDir.replace("~", System.getProperty("user.home"));
                resourceDir = p.resolve(resourceDir).normalize().toString();
                if (!resourceDir.endsWith(FILE_SEPARATOR)) {
                    resourceDir += FILE_SEPARATOR;
                }
                File dir = new File(resourceDir);
                extractedDirectories.add(resourceDir);
            }
        }
        return extractedDirectories;
    }
    
    
    /**
     * Get the Distribution resources directory.
     *
     * @return The resources directory in the distribution.
     */
    static String getDistributionResourcesDirectory() {
        return getDistributionResourcesDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionResourcesDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_RESOURCES_DIRECTORY;
    }

    /**
     * Get the lib directory in the distribution.
     *
     * @return the lib files directory.
     *
     */
    static String getDistributionLibDirectory() {
        return getDistributionLibDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionLibDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_LIB_DIRECTORY;
    }

    /**
     * Get the jni directory in the distribution.
     *
     * @return the jni files directory.
     *
     */
    static String getDistributionJniDirectory() {
        return getDistributionJniDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionJniDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_JNI_DIRECTORY;
    }

    /**
     * Get the jar files directory in the distribution.
     *
     * @return the jar files directory.
     *
     */
    static String getDistributionJarFilesDirectory() {
        return getDistributionJarFilesDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionJarFilesDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_JARS_DIRECTORY;
    }

    /**
     * Get the root directory of the CCS distribution from which the Bootstrap
     * code has been launched.
     *
     * @return The root directory of the CCS distribution. It ends with a
     * file.separator.
     */
    static synchronized String getCCSDistributionRootDirectory() {
        if (ccsDistributionRootDirectory == null) {
            ccsDistributionRootDirectory = getDirectoryForClass(Bootstrap.getLoaderClass());
        }
        return ccsDistributionRootDirectory;
    }

    /**
     * This is to be used only in test environment.
     *
     * @param clazz
     * @return
     */
    static String getDirectoryForClass(Class clazz) {
        URL location = clazz.getProtectionDomain().getCodeSource().getLocation();
        String tmpCcsDistributionRootDirectory = "";
        try {
            Path path = Paths.get(location.toURI());
            String sourceCodeLocation = path.toString();
            if (sourceCodeLocation.endsWith(".jar")) {
                tmpCcsDistributionRootDirectory = sourceCodeLocation.substring(0, sourceCodeLocation.lastIndexOf(FILE_SEPARATOR) + 1);
            } else if (path.toFile().isDirectory()) {
                tmpCcsDistributionRootDirectory = sourceCodeLocation + FILE_SEPARATOR;
            } else {
                throw new RuntimeException("Could not process souce code location " + sourceCodeLocation + " It is neither a directory nor a jar file.");
            }
        } catch (URISyntaxException x) {
            throw new RuntimeException("Could not process souce code location ", x);
        }

        if (tmpCcsDistributionRootDirectory.endsWith(DISTRIBUTION_JARS_DIRECTORY)) {
            tmpCcsDistributionRootDirectory = tmpCcsDistributionRootDirectory.replace(DISTRIBUTION_JARS_DIRECTORY, "");
//            throw new RuntimeException("This executable is not being run from a CCS distribution directory. "+ccsDistributionRootDirectory);
        }
        return tmpCcsDistributionRootDirectory;
    }

    /**
     * Get the list of available applications for the current Bootstrap
     * configuration.
     *
     * @return The list of application names.
     *
     */
    synchronized static List<String> getBootstrapListOfApplications() {
        if (listOfApplications == null) {
            listOfApplications = new ArrayList<>();
            List<String> allPropertiesInResources = new ArrayList<>();
            for (ResourceDirectory dir : getBootstrapResourcesTree().getResourceDirectoryList()) {
                File resourceDirectoryFile = new File(dir.getResouceDirectoryPath());
                if (resourceDirectoryFile.exists() && resourceDirectoryFile.isDirectory()) {
                    String[] listOfResources = resourceDirectoryFile.list();
                    for (String resource : listOfResources) {
                        //Application definition file can have file extension of either "app" or "properties"
                        //LSSTCCS-1647 Remove ".properties" from the search for applications.
                        if (resource.endsWith(".properties") || resource.endsWith(".app")) {
                            String appName = resource.replace(".properties", "");
                            appName = appName.replace(".app", "");
                            allPropertiesInResources.add(appName);
                        }

                    }
                }
            }

            for (String propertyFile : allPropertiesInResources) {
                //LSSTCCS-1647 method getApplicationDefinitionFile will no longer have to
                //load files with extension ".properties"
                Properties props = ResourcesUtils.getApplicationDefinitionFile(getBootstrapResourcesTree(), propertyFile);
                String mainClass = props.getProperty(APPLICATION_MAINCLASS_PROPERTY, "");
                String extendsApp = props.getProperty(APPLICATION_EXTENDS_PROPERTY, "");
                //Application definition file can have file extension of either "app" or "properties"
                //LSSTCCS-1647 Remove ".properties" replacement below
                String applicationName = propertyFile.replace(".properties", "");
                applicationName = applicationName.replace(".app", "");
                if ((!mainClass.isEmpty() || !extendsApp.isEmpty()) && !listOfApplications.contains(applicationName)) {
                    if ( propertyFile.endsWith(".properties") ) {                   
                        LOG.log(Level.WARNING, "Application definition files with extension \".properties\" are being discontinued; only extension \".app\" will be supported.\nPlease rename {0} to {1}", new Object[]{propertyFile, propertyFile.replace(".properties",".app")});
                    }
                    listOfApplications.add(applicationName);
                }
            }
        }
        return listOfApplications;
    }

    /**
     * Get the application definition properties file for a given application
     * name.
     *
     * @param application The name of the application.
     * @return The Properties that define the application.
     */
    static Properties getApplicationDefinitionProperties(String application) {
//        if (!application.endsWith(".properties")) {
//            application = application += ".properties";
//        }
        return BootstrapResourceUtils.getBootstrapProperties(application);
    }

    /**
     * Parse a user provided property to match environment variables specified
     * in the form ${ENV_VARIABLE}
     *
     */
    static String parseProperty(String inputProperty) {
        String outProperty = inputProperty;
        Matcher m = envVar_pattern.matcher(outProperty);
        if (m.matches()) {
            String envVarValue = System.getenv(m.group(2));
            if (envVarValue != null) {
                outProperty = outProperty.replace(m.group(1), envVarValue);
            } else {
                if (Bootstrap.isBootstrapEnvironment()) {
                    System.err.println("[WARNING] Environment variable " + m.group(2) + " is not defined.");
                }
            }
        }
        return outProperty;
    }

}
