package org.lsst.ccs.bootstrap;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.ListIterator;
import java.util.Properties;
import java.util.Set;


/**
 * Utility methods for loading Bootstrap Resources.
 * <p>
 * The utility methods provided in this class are meant to be used throughout the CCS environment
 * to load resources in the Bootstrap environment. 
 * <p>
 * For a description on how to control the Bootstrap environment please refer to the
 * following <a href="https://confluence.slac.stanford.edu/x/-upFC">manual</a>.
 *
 * @author The LSST CCS Team
 */
public abstract class BootstrapResourceUtils {
    
    
    /**
     * Return the Bootstrap System Properties object.
     * <p>
     * This function builds a Properties object by searching and chaining properties
     * according to the following search criteria
     * <ul>
     * <li><b>System Properties</b>: System Properties are the first Properties
     * to be added to the merge chain.</li>
     * <li><b>Bootstrap Resources search</b> : The Bootstrap Resources directories
     * (defined by the environment variable CCS_RESOURCE_PATH) are searched in the
     * reversed order. In each directory properties are merged by searching for
     * application and global properties files (these files are loaded from the
     * file system). </li>
     * <li><b>Command line </b>: Properties specified from the command line with
     * the -D option are added to the top of the Properties chain.</li>
     * </ul>
     *
     * @return The merged Properties object.
     */
    public static Properties getBootstrapSystemProperties() {
        if( Bootstrap.verbose() && ! Bootstrap.isQuiet() ) {
            System.out.println("*** Getting Bootstrap System Properties");
        }
        return buildBootstrapProperties(null, null, true);
    }

    /**
     * Utility function to find the top most property file in the user provided
     * resource directories (as specified in the CCS_RESOURCE_PATH environment
     * variable). If the CCS_RESOURCE_PATH environment variable is not specified
     * null will be returned. If specified the first property file will be returned
     * scanning the resource directories in CCS_RESOURCE_PATH left to right.
     *
     * @param propertyFileName The name of the property file to get
     * @return   The full path of the property file.
     * 
     */
    public static String getPathOfPropertiesFileInUserResourceDirectories(String propertyFileName) {
        
        if ( ! propertyFileName.endsWith(".properties") ) {
            propertyFileName += ".properties";
        }
        
        ResourcesTree tree = BootstrapUtils.getBootstrapResourcesTree();
        
        ListIterator<ResourceDirectory> resourceDirs = tree.getResourceDirectoryList().listIterator();
        while (resourceDirs.hasNext() ) {
            ResourceDirectory resourceDir = resourceDirs.next();
            if ( ! resourceDir.isDistributionDir() ) {
                File resourceFile = new File(resourceDir.getResouceDirectoryPath()+propertyFileName);
                if ( resourceFile.exists() ) {
                    return resourceDir.getResouceDirectoryPath()+propertyFileName;
                }
            }
        }
        
        return null;
    }

    /**
     * Utility function to get the top User defined Resource Directory.
     * This is the left-most directory defined in environment variable CCS_RESOURCE_PATH. 
     * If not defined, null is returned.
     * @return the top user defined Resource directory. Null if not defined.
     */
    
    public static String getTopUserResourceDirectory() {
        ResourcesTree tree = BootstrapUtils.getBootstrapResourcesTree();
        
        ListIterator<ResourceDirectory> resourceDirs = tree.getResourceDirectoryList().listIterator();
        while (resourceDirs.hasNext() ) {
            ResourceDirectory resourceDir = resourceDirs.next();
            if ( ! resourceDir.isDistributionDir() ) {
                return resourceDir.getResouceDirectoryPath();
            }
        }        
        return null;
    }
    
    
    /**
     * Utility function to fetch Properties in the Bootstrap environment.
     * <p>
     * This function builds a Properties object by searching and chaining properties
     * according to the following search criteria
     * <ul>
     * <li><b>Classpath search</b>: getResourceAsStream is invoked on the provided class.
     * The method is first invoked for fileName. If not input stream is found it is then invoked for
     * "/"+fileName</li>
     * <li><b>System Properties</b>: if the parameter includeSystem is set to true, System
     * Properties are added to the chain</li>
     * <li><b>Bootstrap Resources search</b>:
     * The Bootstrap Resources directories (defined by the environment variable
     * CCS_RESOURCE_PATH) are searched in the reversed order. In each directory
     * properties are merged by searching for application, global and topic
     * (defined by fileName) properties files (these files are loaded from the
     * file system). If fileName is a path to a properties file, two searches
     * are performed for each resource directory: the first one is for the fully
     * qualified path; if this yields no result, the properties file is then
     * searched at the root of each resource directory.</li>
     * <li><b>Command line</b>:
     * Properties specified from the command line with the -D option are added
     * to the top of the Properties chain.</li>
     * </ul>
     *
     * @param fileName The name of the properties file to load. This can be a path.
     * @param clazz The class on which getResourceAsStream is invoked for the classpath search. If null BootstrapResourceUtils.class is used.
     * @return The merged Properties object.
     *
     */
    public static Properties getBootstrapProperties(String fileName, Class clazz) {
        if (clazz == null) {
            clazz = Bootstrap.getLoaderClass();
        }
        if ( Bootstrap.verbose() && ! Bootstrap.isQuiet() ) {
            System.out.println("*** Looking for Properties "+fileName);
        }
        InputStream in = getResourceFromClassPath(clazz, fileName);  
        return buildBootstrapProperties(in, fileName, false);
    }

    /**
     * Utility function to fetch Properties in the Bootstrap environment.
     * <p>
     * This function is similar to {@link #getBootstrapProperties(java.lang.String, boolean, java.lang.Class) }.
     * In this case the Class object is set to null and it defaults to BootstrapResourceUtils.class.
     * 
     * @see #getBootstrapProperties(java.lang.String, boolean, java.lang.Class)
     * @param fileName The name of the properties file to load. This can be a path.
     * @return The merged Properties object.
     * 
     */
    public static Properties getBootstrapProperties(String fileName) {
        return getBootstrapProperties(fileName, Bootstrap.getLoaderClass());
    }
   
    /**
     * Utility function to fetch Properties in the Bootstrap environment.
     * <p>
     * This function is similar to {@link #getBootstrapProperties(java.lang.String, boolean, java.lang.Class) } but
     * for the fact that a ClassLoader is used for the <b>Classpath search</b>, rather than a Class.
     *
     * @see #getBootstrapProperties(java.lang.String, boolean, java.lang.Class) 
     * @param fileName The name of the properties file to load. This can be a path.
     * @param classLoader The classLoader on which getResourceAsStream is invoked for the classpath search.
     * @return The merged Properties object.
     *
     */
    static Properties getBootstrapProperties(String fileName, ClassLoader classLoader) {
        if ( Bootstrap.verbose() && ! Bootstrap.isQuiet()) {
            System.out.println("*** Looking for Properties "+fileName);
        }
        InputStream in = getResourceFromClassPath(classLoader, fileName);
        return buildBootstrapProperties(in, fileName, false);
    }

    private static Properties buildBootstrapProperties(InputStream in, String fileName, boolean includeSystem) {
        ResourcesTreeProperties props = null;

        if (in != null && fileName != null) {
            props = new ResourcesTreeProperties("ClassPath " + fileName, null, null);
            try {
                props.load(in);
            } catch (IOException ioe) {
                throw new RuntimeException("Problem loading properties from classpath", ioe);
            }
        }

        String shortName = getShortNameForFileName(fileName);

        String applicationLevel = Bootstrap.getBootstrapApplication();
        String topicLevel = fileName;
        
        if (BootstrapUtils.getBootstrapListOfApplications().contains(shortName)) {
            applicationLevel = shortName;
            topicLevel = null;
        }

        props = (ResourcesTreeProperties) ResourcesUtils.getMergedProperties(BootstrapUtils.getBootstrapResourcesTree(), new String[]{topicLevel, "ccsGlobal", applicationLevel}, includeSystem, props);
        if (!Bootstrap.getCmdLineProperties().isEmpty()) {
            if ( Bootstrap.verbose() && ! Bootstrap.isQuiet() ) {
                System.out.println("*** Adding command line properties to properties chain.");
            }
            ResourcesTreeProperties cmdLinePros = new ResourcesTreeProperties("Command Line Properties", null, props);
            cmdLinePros.putAll(Bootstrap.getCmdLineProperties());
            props = cmdLinePros;
        }
        
        //If it's a test environment add all System properties that start with "bootstrap.test."
        if ( "true".equals(System.getProperty("org.lsst.ccs.bootstrap.test") ) ) {
            Properties testProperties = System.getProperties();
            for ( String key : testProperties.stringPropertyNames() ) {
                if ( key.startsWith("bootstrap.test.") ) {
                    props.put(key.replace("bootstrap.test.",""), testProperties.getProperty(key));
                }
            }
            
        }
        if ( Bootstrap.verbose() && ! Bootstrap.isQuiet() ) {
            ResourcesUtils.printProperties(props);
        }
        return props;

    }


    /**
     * Get an InputStream corresponding to the Bootstrap Properties object.
     * <p>
     * This function is similar to {@link #getBootstrapProperties(java.lang.String, boolean, java.lang.Class) }.
     * Rather than a Properties object it returns the InputStream corresponsing to 
     * that Properties object. This is done by invoking {@link java.util.Properties#store(java.io.OutputStream, java.lang.String) }
     * on a ByteArrayOutputStream and then returning an ByteArrayInputStream.
     * <p>
     *
     * @see BootstrapResourceUtils#getBootstrapProperties(java.lang.String, boolean, java.lang.Class)
     * @param fileName The name of the properties file to load. This can be a path.
     * @param clazz The class on which getResourceAsStream is invoked for the classpath search. If null BootstrapResourceUtils.class is used.
     * @return
     */
    public static InputStream getBootstrapPropertiesInputStream(String fileName, Class clazz) {
        Properties props = getBootstrapProperties(fileName, clazz);
        if (props == null) {
            return null;
        }

        Properties outProps = props;
        if ( props instanceof ResourcesTreeProperties ) {
            outProps = new Properties();
            ((ResourcesTreeProperties)props).copyProperties(outProps);
        } 
        
        InputStream is = null;
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        try {
            outProps.store(output, null);
            is = new ByteArrayInputStream(output.toByteArray());
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }

        return is;
    }

    /**
     * Utility functions to look for resources (excluding Properties) in the Bootstrap environment.
     * <p>
     * This function will search for the desired resource following the search criteria described below. 
     * It will return an InputStream for the first resource found. 
     * <p>
     * Search Criteria:
     * <ul>
     * <li><b>Bootstrap Resources search</b>: The Bootstrap Resources directories (defined by the
     * environment variable CCS_RESOURCE_PATH) are searched in the defined
     * order. If fileName is a path to a resource, two searches are performed
     * for each resource directory: the first one is for the fully qualified
     * path; if this yields no result, the resource file is then searched at the
     * root of each resource directory.</li>
     * <li><b>Classpath search</b>:
     * getResourceAsStrean is invoked on the provided clazz. The method is first invoked for
     * fileName. If not input stream is found it is then invoked for "/"+fileName</li>
     * </ul>
     * <p>
     * For Properties refer to {@link #getBootstrapProperties(java.lang.String, boolean,java.lang.Class).
     * 
     * @param resourceName The name of the resource. This can be a path.
     * @param clazz The class on which getResourceAsStream is invoked for the
     * classpath search. If null BootstrapResourceUtils.class is used.
     * @return The InputStream of the first found resource.
     *
     */
    public static InputStream getBootstrapResource(String resourceName, Class clazz) {
        if (clazz == null) {
            clazz = Bootstrap.getLoaderClass();
        }
        if ( Bootstrap.verbose() && ! Bootstrap.isQuiet() ) {
            System.out.println("*** Looking for resource "+resourceName+" for Class "+clazz);
        }

        InputStream in = null;

        try {
            in = ResourcesUtils.getResourceFromResourceTree(BootstrapUtils.getBootstrapResourcesTree(), resourceName);
        } catch (FileNotFoundException fnfe) {
            throw new RuntimeException(fnfe);
        }

        if (in != null) {
            return in;
        } else {
            return getResourceFromClassPath(clazz, resourceName);
        }

    }

    /**
     * Utility functions to look for resources (excluding Properties) in the Bootstrap environment.
     * <p>
     * Internally it invokes {@link getBootstrapResource(java.lang.String,java.lang.Class)} with Class
     * set to BootstrapResourceUtils.class.
     * 
     * @see #getBootstrapResource(java.lang.String, java.lang.Class) 
     * @param resourceName The name of the resource. This can be a path.
     * @return The InputStream of the first found resource.
     *
     */
    public static InputStream getBootstrapResource(String resourceName) {
        return getBootstrapResource(resourceName, Bootstrap.getLoaderClass());
    }

    
    /**
     * Utility functions to look for resources (excluding properties)
     * in the Bootstrap environment.
     * <p>
     * This function is similar to {@link #getBootstrapResource(java.lang.String, java.lang.Class) } but
     * for the fact that a ClassLoader is used for the <b>Classpath search</b>, rather than a Class.
     *
     * @see #getBootstrapResource(java.lang.String, java.lang.Class) 
     * @param resourceName The name of the resource. This can be a path.
     * @param classLoader The classLoader on which getResourceAsStream is invoked for the classpath search. 
     * @return The InputStream of the first found resource.
     *
     */
    static InputStream getBootstrapResource(String resourceName, ClassLoader classLoader) {
        if ( Bootstrap.verbose() && ! Bootstrap.isQuiet() ) {
            System.out.println("*** Looking for resource "+resourceName+" for ClassLoader "+classLoader);
        }

        InputStream in = null;

        try {
            in = ResourcesUtils.getResourceFromResourceTree(BootstrapUtils.getBootstrapResourcesTree(), resourceName);
        } catch (FileNotFoundException fnfe) {
            throw new RuntimeException(fnfe);
        }

        if (in != null) {
            return in;
        } else {
            return getResourceFromClassPath(classLoader, resourceName);
        }

    }

    
    /**
     * Utility method to fetch all the keys in a properties file, including its parent, to loop over them.
     * <p>
     * This method keeps into account if a Properties object has a parent and goes down
     * the parent chain to fetch all the keys.
     * @param props The Properties object
     * @return The Set<Object> of all the keys in the Properties object.
     */
    public static Set<Object> getAllKeysInProperties(Properties props) {
        Set<Object> keySet = new HashSet<>();
        if (props != null) {
            ResourcesUtils.loadKeySetForProperties(props, keySet);
        }
        return keySet;
    }
    

    private static InputStream getResourceFromClassPath(Class clazz, String resourceName) {
        InputStream in = clazz.getResourceAsStream(resourceName);
        if (in == null && !resourceName.startsWith(BootstrapUtils.FILE_SEPARATOR)) {
            resourceName = BootstrapUtils.FILE_SEPARATOR + resourceName;
            in = clazz.getResourceAsStream(resourceName);
        }
        if ( (Bootstrap.verbose() && in != null) && ! Bootstrap.isQuiet() ) {
            System.out.println("*** Found resource "+resourceName+" in classpath for Class "+clazz);
        }
        return in;
    }

    private static InputStream getResourceFromClassPath(ClassLoader classLoader, String resourceName) {
        InputStream in = classLoader.getResourceAsStream(resourceName);

        if (in == null && !resourceName.startsWith(BootstrapUtils.FILE_SEPARATOR)) {
            resourceName = BootstrapUtils.FILE_SEPARATOR + resourceName;
            in = classLoader.getResourceAsStream(resourceName);
        }
        if ( (Bootstrap.verbose() && in != null) && ! Bootstrap.isQuiet() ) {
            System.out.println("*** Found resource "+resourceName+" in classpath for ClassLoader "+classLoader);
        }
        return in;
    }

    private static String getShortNameForFileName(String fileName) {
        if (fileName == null) {
            return null;
        }
        String bootstrapPropertyName = fileName;
        if (fileName.contains(BootstrapUtils.FILE_SEPARATOR)) {
            bootstrapPropertyName = fileName.substring(fileName.lastIndexOf(BootstrapUtils.FILE_SEPARATOR) + 1);
        }
        String shortName = bootstrapPropertyName.replace(".properties", "");
        return shortName;
    }

}
